using System;
using Microsoft.SPOT;
using F75XDeviceApp.Common;
using System.IO;
using Common;

namespace F75XDeviceApp
{
    /**
     * Model Builder stores...
     * 
     * 1) Measurement files
     * 
     * 1) GetModelSpectra is called.  
     * 
     * 2) SetPrediction is called during each use of the model to predict/validate on the device and in App Builder.
     * 
     * 3) SaveTrainingData is run before building the model in AppBuilder
     * 
     * 4) BuildMetaData is run in AppBuilder
     * 
     * 5) LoadMetaData is run by App on device
     * 
     * 6) GetTrainingDataRecord is run by app on device optionally
     * 
     */
    public class PLSModel : Model
    {

        public float[] RegressionCoefficients = new float[0];
        public float Intercept = 0f;
  
        public override void BuildSpectra(Measurement aMeasurement)
        {
            AppMeasurement newMeasurement = (AppMeasurement)aMeasurement;

            float binSizeInNM = Wavelengths[1] - Wavelengths[0];

            float[] modelSpectra = new float[newMeasurement.SpectrometerSensorCount];
            
            if (Name.IndexOf("_BS") > -1)
            {
            	SetBSExampleSpecra(newMeasurement, modelSpectra);
            }
            else 
            {            
            	SetReflectanceSpecra(newMeasurement, modelSpectra);
            	SetAbsorbanceSpectra(newMeasurement, modelSpectra);
            	SetSavGol2DSpectra(newMeasurement, modelSpectra);
	    }

            modelSpectra = Core.Utils.SpectraUtils.GetInterpolatedSpectra(
                                            newMeasurement.WavelengthInterpolationC0,
                                            newMeasurement.WavelengthInterpolationC1,
                                            newMeasurement.WavelengthInterpolationC2,
                                            newMeasurement.WavelengthInterpolationC3,
                                            newMeasurement.WavelengthInterpolationC4,
                                            newMeasurement.WavelengthInterpolationC5,
                                            modelSpectra, binSizeInNM);

            if (Spectra.Length != Wavelengths.Length)
            {
                Spectra = new float[Wavelengths.Length];
            }
            else
            {
                Array.Clear(Spectra, 0, Spectra.Length);
            }

            for (int i = 0; i < Wavelengths.Length; i += 1)
            {
                int binIndex = Array.IndexOf(Core.Utils.SpectraUtils.InterpolatedWavelengths, Wavelengths[i]);

                if (binIndex > -1)
                {
                    Spectra[i] = modelSpectra[binIndex] * 1.5f;
                }
            }

            if (F75XDeviceApp.Common.Runtime.IsNETMF)
            {
                LoadTraningSetStats();
            }

            float[] averageSpectra = GetTrainingDataStats("Average");
            float[] stDev = GetTrainingDataStats("MeanStandardDeviation");

            if (MeanCentering && ((averageSpectra != null) && (Spectra.Length == averageSpectra.Length)))
            {
                for (int i = 0; i < Wavelengths.Length; i += 1)
                {
                    Spectra[i] = (Spectra[i] - averageSpectra[i]);
                }
            }
            else if (ZMUV && (((averageSpectra != null) && (Spectra.Length == averageSpectra.Length)) && ((stDev != null) && (Spectra.Length == stDev.Length))))
            {
                for (int i = 0; i < Wavelengths.Length; i += 1)
                {
                    Spectra[i] = (Spectra[i] - averageSpectra[i]) / stDev[i];
                }
            }

        }

        public bool MeanCentering = false;
        public bool ZMUV = false;

        public override void SetPrediction(Measurement aMeasurement)
        {
            float recoSum = 0f;

            if (RegressionCoefficients.Length == Spectra.Length)
            {
                for (int i = 0; i < RegressionCoefficients.Length; i += 1)
                {
                    recoSum += Spectra[i] * RegressionCoefficients[i];
                }
            }


            //This code will only run on the device...
            if (F75XDeviceApp.Common.Runtime.IsNETMF)
            {
                /* FileStream fs = System.IO.File.OpenWrite(@"\SD\Debug_Spectra.csv");
                 fs.Seek(0, SeekOrigin.End);
                 StreamWriter sw = new StreamWriter(fs);

                 for (int i = 0; i < Spectra.Length; i += 1)
                 {
                     sw.Write(Spectra[i].ToString() + "\t");
                 }
                 sw.Write("\r\n");
                 sw.Flush();
                 fs.Flush();
                 fs.Close();*/
            }

            PredictionValue = Intercept + recoSum;

            base.SetPrediction(aMeasurement);
        }

	public virtual void SetBSExampleSpecra(AppMeasurement aMeasurement, float[] aModelSpectra)
        {
            for (int i = 0; i < aModelSpectra.Length; i += 1)
            {
                aModelSpectra[i] = aMeasurement.LightSpecimenSpectra[i] / aMeasurement.SampleIntegrationTime;

            }
        }


        public virtual void SetReflectanceSpecra(AppMeasurement aMeasurement, float[] aModelSpectra)
        {
            for (int i = 0; i < aModelSpectra.Length; i += 1)
            {
            	float a = ((aMeasurement.LightSpecimenSpectra[i] - aMeasurement.DarkSpecimenSpectra[i]) / aMeasurement.SampleIntegrationTime);
            	
            	float b = ((aMeasurement.LightReferenceSpectra[i] - aMeasurement.DarkReferenceSpectra[i]) / aMeasurement.ReferenceIntegrationTime);
            	
            	if ((a == 0) || (b == 0))
            	{
            		aModelSpectra[i] = 0f;
            	}
            	else 
            	{ 
			aModelSpectra[i] = (
						    a
						    /
						    b
				    );
            	}

            }
        }

        protected virtual void SetAbsorbanceSpectra(AppMeasurement aMeasurement, float[] aModelSpectra)
        {
            float currentValue = 0;
            for (int i = 0; i < aModelSpectra.Length; i += 1)
            {
                currentValue = aModelSpectra[i];
                if (currentValue != 0)
                {
                    aModelSpectra[i] = (float)System.Math.Log10(1f / currentValue);
                }
            }
        }

        protected static float[] smoothedDerivativeAbsorbanceSpectra = new float[0];

        /**
         * Taken from convolution coefficients for polynomials based on Savitzky, A.; Golay, M.J.E. (1964). 
         * "Smoothing and Differentiation of Data by Simplified Least Squares Procedures".
         */
        protected virtual void SetSavGol2DSpectra(AppMeasurement aMeasurement, float[] aModelSpectra)
        {
            if (smoothedDerivativeAbsorbanceSpectra.Length != aModelSpectra.Length)
            {
                smoothedDerivativeAbsorbanceSpectra = new float[aModelSpectra.Length];
            }
            else
            {
                Array.Clear(smoothedDerivativeAbsorbanceSpectra, 0, smoothedDerivativeAbsorbanceSpectra.Length);
            }

            /*  float[] savitzkyGolayCoefficients = new float[] { 0.0174825174825175f,
                                                              0.00699300699300699f,
                                                              -0.00116550116550117f,
                                                              -0.00699300699300699f,
                                                              -0.0104895104895105f,
                                                              -0.0116550116550117f,
                                                              -0.0104895104895105f,
                                                              -0.00699300699300699f,
                                                              -0.00116550116550117f,
                                                              0.00699300699300699f,
                                                              0.0174825174825175f };*/

            float[] savitzkyGolayCoefficients = new float[] { 
                                                            0.030303030303030303f, // -4 
                                                            0.007575757575757575f, // -3 
                                                           -0.00865800865800f, // -2 
                                                            -0.0183982684f, // -1 
                                                            -0.021645021645f,// center 
                                                            -0.0183982684f, // +1 
                                                            -0.00865800865800f, // +2 
                                                            0.007575757575757575f, // +3 
                                                            0.030303030303030303f // +4 
                                                            };

            for (int pixel = 0; pixel < 4; ++pixel)
            {
                smoothedDerivativeAbsorbanceSpectra[pixel] = 0f;
            }

            for (int pixel = 4; pixel < smoothedDerivativeAbsorbanceSpectra.Length - 4; ++pixel)
            {
                smoothedDerivativeAbsorbanceSpectra[pixel] += (
                                                                (
                                                                    (aModelSpectra[pixel - 4] * savitzkyGolayCoefficients[0]) +
                                                                    (aModelSpectra[pixel - 3] * savitzkyGolayCoefficients[1]) +
                                                                    (aModelSpectra[pixel - 2] * savitzkyGolayCoefficients[2]) +
                                                                    (aModelSpectra[pixel - 1] * savitzkyGolayCoefficients[3]) +
                                                                    (aModelSpectra[pixel] * savitzkyGolayCoefficients[4]) +
                                                                    (aModelSpectra[pixel + 1] * savitzkyGolayCoefficients[5]) +
                                                                    (aModelSpectra[pixel + 2] * savitzkyGolayCoefficients[6]) +
                                                                    (aModelSpectra[pixel + 3] * savitzkyGolayCoefficients[7]) +
                                                                    (aModelSpectra[pixel + 4] * savitzkyGolayCoefficients[8])
                                                                    )
                                                                );

            }


            for (int pixel = smoothedDerivativeAbsorbanceSpectra.Length - 4; pixel < smoothedDerivativeAbsorbanceSpectra.Length; ++pixel)
            {
                smoothedDerivativeAbsorbanceSpectra[pixel] = 0f;
            }

            for (int pixel = 0; pixel < smoothedDerivativeAbsorbanceSpectra.Length; ++pixel)
            {
                smoothedDerivativeAbsorbanceSpectra[pixel] = smoothedDerivativeAbsorbanceSpectra[pixel];

                if (Double.IsInfinity(smoothedDerivativeAbsorbanceSpectra[pixel]) ||
                        Double.IsNaN(smoothedDerivativeAbsorbanceSpectra[pixel]) ||
                        Double.IsNegativeInfinity(smoothedDerivativeAbsorbanceSpectra[pixel]) ||
                        Double.IsPositiveInfinity(smoothedDerivativeAbsorbanceSpectra[pixel]))
                {
                    smoothedDerivativeAbsorbanceSpectra[pixel] = 0;
                }
            }

            Array.Copy(smoothedDerivativeAbsorbanceSpectra, aModelSpectra, aModelSpectra.Length);

            /*for (int pixel = 0; pixel < smoothedDerivativeAbsorbanceSpectra.Length; ++pixel)
            {
                aModelSpectra[pixel] = smoothedDerivativeAbsorbanceSpectra[pixel];
            }*/
        }


        /**
         * Taken from convolution coefficients for polynomials based on Savitzky, A.; Golay, M.J.E. (1964). 
         * "Smoothing and Differentiation of Data by Simplified Least Squares Procedures".
         */
        protected virtual void SetSavGol2DSpectraNew(AppMeasurement aMeasurement, float[] aModelSpectra)
        {
            if (smoothedDerivativeAbsorbanceSpectra.Length != aModelSpectra.Length)
            {
                smoothedDerivativeAbsorbanceSpectra = new float[aModelSpectra.Length];
            }
            else
            {
                Array.Clear(smoothedDerivativeAbsorbanceSpectra, 0, smoothedDerivativeAbsorbanceSpectra.Length);
            }

            float[] savitzkyGolayCoefficients = new float[] { 0.0174825174825175f,
                                                            0.00699300699300699f,
                                                            -0.00116550116550117f,
                                                            -0.00699300699300699f,
                                                            -0.0104895104895105f,
                                                            -0.0116550116550117f,
                                                            -0.0104895104895105f,
                                                            -0.00699300699300699f,
                                                            -0.00116550116550117f,
                                                            0.00699300699300699f,
                                                            0.0174825174825175f };


            for (int pixel = 0; pixel < 5; ++pixel)
            {
                smoothedDerivativeAbsorbanceSpectra[pixel] = 0f;
            }

            for (int pixel = 5; pixel < aModelSpectra.Length - 5; ++pixel)
            {
                smoothedDerivativeAbsorbanceSpectra[pixel] += (
                                                                (
                                                                    (aModelSpectra[pixel - 5] * savitzkyGolayCoefficients[0]) +
                                                                    (aModelSpectra[pixel - 4] * savitzkyGolayCoefficients[1]) +
                                                                    (aModelSpectra[pixel - 3] * savitzkyGolayCoefficients[2]) +
                                                                    (aModelSpectra[pixel - 2] * savitzkyGolayCoefficients[3]) +
                                                                    (aModelSpectra[pixel - 1] * savitzkyGolayCoefficients[4]) +
                                                                    (aModelSpectra[pixel] * savitzkyGolayCoefficients[5]) +
                                                                    (aModelSpectra[pixel + 1] * savitzkyGolayCoefficients[6]) +
                                                                    (aModelSpectra[pixel + 2] * savitzkyGolayCoefficients[7]) +
                                                                    (aModelSpectra[pixel + 3] * savitzkyGolayCoefficients[8]) +
                                                                    (aModelSpectra[pixel + 4] * savitzkyGolayCoefficients[9]) +
                                                                    (aModelSpectra[pixel + 5] * savitzkyGolayCoefficients[10])
                                                                    )
                                                                );

            }


            for (int pixel = aModelSpectra.Length - 5; pixel < aModelSpectra.Length; ++pixel)
            {
                smoothedDerivativeAbsorbanceSpectra[pixel] = 0f;
            }

            for (int pixel = 0; pixel < aModelSpectra.Length; ++pixel)
            {
                smoothedDerivativeAbsorbanceSpectra[pixel] = smoothedDerivativeAbsorbanceSpectra[pixel];

                if (Double.IsInfinity(smoothedDerivativeAbsorbanceSpectra[pixel]) ||
                        Double.IsNaN(smoothedDerivativeAbsorbanceSpectra[pixel]) ||
                        Double.IsNegativeInfinity(smoothedDerivativeAbsorbanceSpectra[pixel]) ||
                        Double.IsPositiveInfinity(smoothedDerivativeAbsorbanceSpectra[pixel]))
                {
                    smoothedDerivativeAbsorbanceSpectra[pixel] = 0;
                }
            }

            Array.Copy(smoothedDerivativeAbsorbanceSpectra, aModelSpectra, aModelSpectra.Length);

            /*for (int pixel = 0; pixel < smoothedDerivativeAbsorbanceSpectra.Length; ++pixel)
            {
                aModelSpectra[pixel] = smoothedDerivativeAbsorbanceSpectra[pixel];
            }*/
        }

        /*
        public static float[] SplitWavelengths(string aInput, float binSize)
        {
            float[] wavelengths = new float[0]; 

            try
            {
                string wavelengthsString = "";

                string[] segments = aInput.Split(',');

                for (int i = 0; i < segments.Length; i += 1)
                {
                    string[] w2 = segments[i].Split('-');

                    float start = (float)double.Parse(w2[0]);
                    float end = (float)double.Parse(w2[1]);

                    float r = (end - start) % binSize;

                    if (r != 0)
                    {
                        throw new Exception("(end - start) % binSize Err");
                    }

                    for (float w = start; w <= end; w += binSize)
                    {
                        wavelengthsString += w + ",";
                    }

                }

                string[] wavelengthsArray = wavelengthsString.TrimEnd(',').Split(',');
                wavelengths = new float[wavelengthsArray.Length];

                for (int i = 0; i < wavelengthsArray.Length; i += 1)
                {
                    wavelengths[i] = (float)double.Parse(wavelengthsArray[i]);
                }
            }
            catch (Exception ex)
            {
                throw new Exception("Error splitting wavelengths", ex);
            }

            return wavelengths;
        }
        */

        /*
            Data must be loaded in CachedTrainingDataSpectra and CachedTrainingDataReferenceValues 
            -or- 
            TrainingDataFilePath and TrainingDataFileIndex must exist (on device for example)
        */
        public virtual void BuildTraningSetStats()
        {
            TrainingDataStats = new float[2][];

            float[] averageSpectra = new float[Wavelengths.Length];
            float[] meanStandardDeviationSpectra = new float[Wavelengths.Length];

            TrainingDataStats[0] = new float[Wavelengths.Length];
            TrainingDataStats[1] = new float[Wavelengths.Length];

            if (TrainingDataRecordCount > 0)
            {
                for (int w = 0; w < Wavelengths.Length; w += 1)
                {
                    for (int r = 0; r < TrainingDataRecordCount; r += 1)
                    {
                        float[] recordSpectra = GetTrainingDataSpectra(r);
                        averageSpectra[w] += recordSpectra[w];
                    }

                    averageSpectra[w] = averageSpectra[w] / TrainingDataRecordCount;

                    TrainingDataStats[0][w] = averageSpectra[w];
                }

                for (int w = 0; w < Wavelengths.Length; w += 1)
                {
                    double xigma = 0;
                    for (int r = 0; r < TrainingDataRecordCount; r += 1)
                    {
                        float[] recordSpectra = GetTrainingDataSpectra(r);
                        xigma += System.Math.Pow(recordSpectra[w] - averageSpectra[w], 2);
                    }
                    xigma /= TrainingDataRecordCount - 1;

                    TrainingDataStats[1][w] = (float)System.Math.Sqrt(xigma);
                }
            }
        }


        //Only gets called from the device.  These variables are loaded in memory on the PC side.
        public virtual void LoadTraningSetStats()
        {
            try
            {
                TrainingDataStats = new float[TrainingDataStatNames.Length][];

                for (int i = 0; i < TrainingDataStatNames.Length; i += 1)
                {
                    TrainingDataStats[i] = new float[Wavelengths.Length];
                }

                if (!System.IO.File.Exists(TrainingDataStatsFile))
                {
                    BuildTraningSetStats();
                }

                if (System.IO.File.Exists(TrainingDataStatsFile))
                {
                    StreamReader sr = new StreamReader(System.IO.File.OpenRead(TrainingDataStatsFile));

                    string line = null;
                    int lineNumber = 0;
                    while ((line = sr.ReadLine()) != null)
                    {
                        string[] values = line.Split('\t');

                        if (values.Length > 0)
                        {
                            TrainingDataStatNames[lineNumber] = values[0];

                            for (int w = 0; w < Wavelengths.Length; w += 1)
                            {
                                TrainingDataStats[lineNumber][w] = (float)double.Parse(values[w + 1]);
                            }
                        }
                        lineNumber += 1;
                    }
                }
            }
            catch (Exception ex)
            {
                throw new Exception("Error loading training set stats", ex);
            }
        }

        public virtual void SaveTraningSetStats()
        {
            if (System.IO.File.Exists(TrainingDataStatsFile))
            {
                System.IO.File.Delete(TrainingDataStatsFile);
            }

            FileStream fs = System.IO.File.OpenWrite(TrainingDataStatsFile);
            StreamWriter sw = new StreamWriter(fs);

            for (int i = 0; i < TrainingDataStatNames.Length; i += 1)
            {
                sw.Write(TrainingDataStatNames[i]);

                for (int w = 0; w < TrainingDataStats[i].Length; w += 1)
                {
                    sw.Write("\t" + TrainingDataStats[i][w].ToString());
                }

                sw.Write("\r\n");
            }
            sw.Flush();
            fs.Flush();
            fs.Close();
        }

        public string[] TrainingDataStatNames = new string[2] { "Average", "MeanStandardDeviation" };
        public float[][] TrainingDataStats = new float[0][];

        public float[] GetTrainingDataStats(string aStatName)
        {
            if ((Array.IndexOf(TrainingDataStatNames, aStatName) == -1) || (TrainingDataStats.Length == 0))
            {
                return null;
            }
            else
            {
                return TrainingDataStats[Array.IndexOf(TrainingDataStatNames, aStatName)];
            }
        }

        //The values below are saved in the .F75XApp file.  The device firmware knows to save a portion of the app file
        //to disk and cache it for later access in the GetTrainingDataSpectra, GetTrainingDataReferenceValue, etc. 
        //functions.
        //
        //The UpdateData syntax:
        //      UpdateData.exe [ConnectionString] [ModelFolder]  
        //
        //      Update data will write out [ModelID].Stats text file. 
        //    
        //      Searches folder for TrainingDataStatsFile.
        //
        public float[][] CachedTrainingDataSpectra = null;
        public float[] CachedTrainingDataReferenceValues = null;

        public float[] GetTrainingDataSpectra(int aRecordNumber)
        {
            if (CachedTrainingDataSpectra != null)
            {
                return CachedTrainingDataSpectra[aRecordNumber];
            }
            else if (aRecordNumber <= TrainingDataRecordCount)
            {
                //Only gets called from the device.  These variables are loaded in memory on the PC side.
                if (System.IO.File.Exists(TrainingDataFilePath))
                {
                    FileStream fs = System.IO.File.OpenRead(TrainingDataFilePath);
                    fs.Seek(4 + (4 * TrainingDataRecordCount) + (aRecordNumber * (4 * Wavelengths.Length)), SeekOrigin.Begin);
                    float[] result = new float[Wavelengths.Length];
                    for (int i = 0; i < result.Length; i += 1)
                    {
                        result[i] = Core.Utils.BitUtils.ToFloat(fs);
                    }
                    fs.Close();
                    return result;
                }
            }

            return null;
        }

        public float GetTrainingDataReferenceValue(int aRecordNumber)
        {
            if (CachedTrainingDataReferenceValues != null)
            {
                return CachedTrainingDataReferenceValues[aRecordNumber];
            }
            else
            {
                if (aRecordNumber > TrainingDataRecordCount)
                {
                    return float.MinValue;
                }

                float result = float.MinValue;

                //Only gets called from the device.  These variables are loaded in memory on the PC side.
                if (System.IO.File.Exists(TrainingDataFilePath))
                {
                    FileStream fs = System.IO.File.OpenRead(TrainingDataFilePath);
                    fs.Seek(4 + (4 * aRecordNumber), SeekOrigin.Begin);
                    byte[] resultBuffer = new byte[4];
                    fs.Read(resultBuffer, 0, resultBuffer.Length);
                    result = Core.Utils.BitUtils.ToFloat(resultBuffer, 0);
                    fs.Close();
                }

                return result;
            }
        }

        public int[] GetTrainingDataRecordNumbers(float aMinReferenceValue, float aMaxReferenceValue)
        {
            float referenceValueBuffer = float.MinValue;
            string recordNumbersBuffer = "";

            for (int i = 0; i < TrainingDataRecordCount; i += 1)
            {
                referenceValueBuffer = GetTrainingDataReferenceValue(i);
                if ((referenceValueBuffer >= aMinReferenceValue) && (referenceValueBuffer <= aMaxReferenceValue))
                {
                    recordNumbersBuffer += i + ",";
                }
            }

            int[] result = null;

            if (recordNumbersBuffer.Length > 0)
            {
                recordNumbersBuffer = recordNumbersBuffer.TrimEnd(',');

                string[] recordNumbersArrayBuffer = recordNumbersBuffer.Split(',');

                result = new int[recordNumbersArrayBuffer.Length];
                for (int i = 0; i < recordNumbersArrayBuffer.Length; i += 1)
                {
                    result[i] = int.Parse(recordNumbersArrayBuffer[i]);
                }
            }

            return result;
        }
    }
}
