using System;
using Microsoft.SPOT; 
using Core.Utils;
using Core.Hardware;
using System.Threading;
using System.IO;
using Core;
using Core.Text;
using Microsoft.SPOT.IO;
using Core.UI;
using Core.Screens;
using System.Text; 
using GHI.OSHW.Hardware.LowLevel;
using Common;

namespace F75XDeviceApp
{
    public class F75XApp : App
    {
        public override Measurement CreateMeasurement()
        {  
            return (Measurement)new AppMeasurement();
        }

	public AppOptions Options = new AppOptions();
	
        /**
        * In the future AppBuilder will generate this code for Customizations.cs
        */
        public F75XApp()
        { 
            BootText = "MB";
            Name = "F75XApp v1.0";
            TextOutputFile = "F75XAppLog.txt";
            ID = new Guid(new byte[] { 0xef, 0xac, 0x31, 0xb4, 0xab, 0x6f, 0x08, 0x48, 0xbe, 0x0c, 0x26, 0x22, 0x73, 0x04, 0x0c, 0x10 });
            Options = new AppOptions();
         
         //GenericMessage.DrawCentered("0:" + Options.DefaultLot, true);
        }

        /**
         * In the future AppBuilder will generate this code for Customizations.cs
         */
        public override void Init()
        {
            base.Init(); 
            
            LotUtils.Root = @"\SD\Apps\F75XApp\Data\";
            
            Options.DataServer = "192.168.1.192";
            Options.DataServerPort = "8080";    
        }

        public override void SaveOptions()
        {
            base.SaveOptions(); 

            Settings.Write(this.Name + ".Options.EstimateSpecimenDarkSpectra", Options.EstimateSpecimenDarkSpectra);
            Settings.Write(this.Name + ".Options.EstimateReferenceDarkSpectra", Options.EstimateReferenceDarkSpectra);
            Settings.Write(this.Name + ".Options.ReferenceIntegrationTime", Options.ReferenceIntegrationTime);
            Settings.Write(this.Name + ".Options.SpecimenIntegrationTime", Options.SampleIntegrationTime);
            Settings.Write(this.Name + ".Options.ReferenceScansToAverage", Options.ReferenceScansToAverage);
            Settings.Write(this.Name + ".Options.SampleScansToAverage", Options.SampleScansToAverage);
            Settings.WriteToSDCard(this.Name + ".DefaultLot", Options.DefaultLot);
            Settings.WriteToSDCard(this.Name + ".DefaultTags", Options.DefaultTags);
            Settings.Write(this.Name + ".Options.DataServer", Options.DataServer);
            Settings.Write(this.Name + ".Options.DataServerPort", Options.DataServerPort);
            Settings.Save();

            Settings.WriteToSDCard(this.Name + ".IncrementationLogic_Lots", Options.IncrementationLogic_Lots);
            Settings.WriteToSDCard(this.Name + ".IncrementationLogic_Maximums", Options.IncrementationLogic_Maximums);

            for (int i = 0; i < Models.Length; i += 1)
            {
                Settings.WriteToSDCard(this.Name + "." + Models[i].PredictionName + ".BiasCorrectionIntercept", Models[i].BiasCorrectionIntercept);
                Settings.WriteToSDCard(this.Name + "." + Models[i].PredictionName + ".BiasCorrectionSlope", Models[i].BiasCorrectionSlope);
            } 
        }

        public override void LoadOptions()
        {
            base.LoadOptions();

            Options.EstimateSpecimenDarkSpectra = Settings.Read(this.Name + ".Options.EstimateSpecimenDarkSpectra", Options.EstimateSpecimenDarkSpectra);
            Options.EstimateReferenceDarkSpectra = Settings.Read(this.Name + ".Options.EstimateReferenceDarkSpectra", Options.EstimateReferenceDarkSpectra);
            Options.ReferenceIntegrationTime = Settings.Read(this.Name + ".Options.ReferenceIntegrationTime", Options.ReferenceIntegrationTime, -1, 99999);
            Options.SampleIntegrationTime = Settings.Read(this.Name + ".Options.SpecimenIntegrationTime", Options.SampleIntegrationTime, -1, 99999);
            Options.ReferenceScansToAverage = Settings.Read(this.Name + ".Options.ReferenceScansToAverage", Options.ReferenceScansToAverage, 1, 255);
            Options.SampleScansToAverage = Settings.Read(this.Name + ".Options.SampleScansToAverage", Options.SampleScansToAverage, 1, 255);
            Options.DefaultLot = Settings.ReadFromSDCard(this.Name + ".DefaultLot", Options.DefaultLot);
            Options.DefaultTags = Settings.ReadFromSDCard(this.Name + ".DefaultTags", Options.DefaultTags);
            Options.DataServer = Settings.Read(this.Name + ".Options.DataServer", Options.DataServer, 4096);
            Options.DataServerPort = Settings.Read(this.Name + ".Options.DataServerPort", Options.DataServerPort, 6);

            Options.IncrementationLogic_Lots = Settings.ReadFromSDCard(this.Name + ".IncrementationLogic_Lots", Options.IncrementationLogic_Lots);
            Options.IncrementationLogic_Maximums = Settings.ReadFromSDCard(this.Name + ".IncrementationLogic_Maximums", Options.IncrementationLogic_Maximums);
           
            for (int i = 0; i < Models.Length; i+=1) 
            {
                Models[i].BiasCorrectionIntercept = Settings.ReadFromSDCard(this.Name + "." + Models[i].PredictionName + ".BiasCorrectionIntercept", 0f);
                Models[i].BiasCorrectionSlope = Settings.ReadFromSDCard(this.Name + "." + Models[i].PredictionName + ".BiasCorrectionSlope", 1f);                 
            } 
        }


        public class AppOptions
        {
            public bool AllowOverwrite = false; //Allows users to overwrite measurements when moving records.
            public  bool EstimateSpecimenDarkSpectra = false; //Indicates to guess the dark current and ignore external light pollution.
            public bool EstimateReferenceDarkSpectra = false; //Indicates to guess the dark current and ignore external light pollution.
            public int ReferenceIntegrationTime = -1; //-1 is automatic, default detection technique is used
            public int SampleIntegrationTime = -1; //-1 is automatic, default detection technique is used
            public int ReferenceScansToAverage = 2; //defaults to 32, allowed from 1 to 254
            public int SampleScansToAverage = 2; //defaults to 32, allowed from 1 to 254 
            public string DefaultLot = "\\Group001\\Specimen001";
            public string DefaultTags = "";
            public string DataServer = "";
            public string DataServerPort = "";

		public string[] IncrementationLogic_Lots = new string[] { "Group", "Sample" };
		public int[] IncrementationLogic_Maximums = new int[] { 5, 3 }; 

        }

        public virtual void ShowNotes()
        {
            Drawing.ClearScreenBuffer(true);

            SmallFont.DrawString(this.Name, 5, 10, false);
            TinyFont.DrawString("ID:" + this.ID, 10, 40, false);
            TinyFont.DrawString("Modified:" + this.BuildDate.ToString("MM/dd/yyyy HH:mm:ss"), 10, 60, false);

            TinyFont.DrawString("Notes:", 10, 95, false);
            TinyFont.DrawString("This model was created by Felix Instruments.  It is used to", 15, 120, false);
            TinyFont.DrawString("collect data to build chemometric models.", 15, 140, false);

            SmallFont.DrawString("Press any key to continue...", 5, 210, false);
            Display.Update(Display.Buffer); 
        }

        public override void DisplayMeasurement(Measurement aMeasurement)
        {
            Drawing.ClearScreenBuffer(true);

            AppMeasurement measurement = (AppMeasurement)aMeasurement; 
             
            TinyFont.DrawString("Saving...", 350, 5, false);

            TinyFont.DrawString("Created: " + measurement.Timestamp.ToString("MM/dd/yyyy HH:mm:ss"), 5, 205, false);
             
            Display.Update(Display.Buffer);
        }

        public override void DisplayMeasurement(string aMeasurementFile, int aRecordNumber, int aRecordCount)
        {
            Drawing.ClearScreenBuffer(true);

            AppMeasurement measurement = new AppMeasurement();
            measurement.FromStream(new MemoryStream(System.IO.File.ReadAllBytes(aMeasurementFile)));

            SmallFont.DrawString(ref Display.Buffer, LotUtils.GetRelativeLotPath(System.IO.Path.GetDirectoryName(aMeasurementFile)), 7, 10, false);

            TinyFont.DrawString(aRecordNumber + " of " + aRecordCount, 350, 5, false);

            TinyFont.DrawString("Created: " + measurement.Timestamp.ToString("MM/dd/yyyy HH:mm:ss"), 5, 205, false);

            if (aRecordCount != 1)
            {
                Drawing.DrawVerticalScrollbar(aRecordCount, aRecordNumber);
            }

            Display.Update(Display.Buffer);
        }

        public override void DisplayLot(string aFullLotPath, int aCurrentLotNo, int aTotalLotCount)
        {
            Drawing.ClearScreenBuffer(true);

            LotUtils.LotStats stats = LotUtils.GetStats(aFullLotPath, 0);

            string lotPath = LotUtils.GetParentLot(LotUtils.GetRelativeLotPath(aFullLotPath));
            if (lotPath == "")
            {
                lotPath = "\\";
            }
            string lotName = System.IO.Path.GetFileName(aFullLotPath);
            string recordCount = stats.RecordCount.ToString();
            string average = StringUtils.Format(stats.Avg, 1);
            string sd = StringUtils.Format(stats.SD, 1);

            lotPath = FontUtils.AddEllipseToEndOfString(FontUtils.FontSize.Small, lotPath, 380);
            lotName = FontUtils.AddEllipseToEndOfString(FontUtils.FontSize.Large, lotName, 380);
            sd = FontUtils.AddEllipseToEndOfString(FontUtils.FontSize.Large, sd, 100);
            average = FontUtils.AddEllipseToEndOfString(FontUtils.FontSize.Large, average, 220);

            LargeFont.DrawString(lotName, 2, 45, false);
            LargeFont.DrawString(recordCount, 2, 147, false);
            LargeFont.DrawString(average, 134, 147, false);
            LargeFont.DrawString(sd, 292, 147, false);

            SmallFont.DrawString(lotPath, 7, 10, false);
            SmallFont.DrawString(Phrases.Common.LOT, 7, 113, false);
            SmallFont.DrawString(Phrases.Common.RECORDS, 7, 213, false);
            SmallFont.DrawString(Phrases.Common.AVG_DM_PERC, 136, 213, false);
            SmallFont.DrawString(Phrases.Common.STDEV, 292, 213, false);

            Drawing.DrawVerticalScrollbar(aTotalLotCount, aCurrentLotNo);

            TinyFont.DrawString(aCurrentLotNo + " of " + aTotalLotCount, 350, 5, false);

            Display.Update(Display.Buffer);
        }
 
 public void DrawProgressRect(int aX, int aY, int aWidth, int aHeight, float aProgress)
 { 
  Drawing.DrawRectangle(aX, aY, (int)System.Math.Max(1, System.Math.Min(aWidth, (aWidth * aProgress))), aHeight, true, Drawing.PixelColor.Black); 
            Display.Update(Display.Buffer);
 }
 
        public override bool CollectRawSpectra(Measurement aGenericMeasurement)
        {
            int pbX = 200;
            int pbY = 217;
            int pbWidth = 184;
            int pbHeight = 12;//4;
            
            //TinyFont.DrawString(ref Display.Buffer, "Measuring...", 150, 200, true);
            
            Core.Hardware.Lamp.On();
            Thread.Sleep(100); //Lamp timeout changed
            
            //UI.Utils.DrawProgressBar("Processing", "Initializing lamp...", Options.DefaultLot.ToUpper(), 0.1f);
            DrawProgressRect(pbX, pbY, pbWidth, pbHeight, 0.1f); 
               
            AppMeasurement newMeasurement = (AppMeasurement)aGenericMeasurement;
 
		newMeasurement.Lot = Options.DefaultLot;
              	newMeasurement.Timestamp = DateTime.Now;
             	newMeasurement.Tags = Options.DefaultTags;
            newMeasurement.ReferenceScansToAverage = Options.ReferenceScansToAverage;
            newMeasurement.SpecimenScansToAverage = Options.SampleScansToAverage;
            newMeasurement.SampleIntegrationTime = Options.SampleIntegrationTime;
            newMeasurement.ReferenceIntegrationTime = Options.ReferenceIntegrationTime;

            int lampSleepTime = 500 - (int)((DateTime.Now.Ticks - Core.Hardware.Lamp.LampOnTime) / TimeSpan.TicksPerMillisecond);
             
            if (lampSleepTime > 0)
            {
                Thread.Sleep(lampSleepTime); //Wait for the light to turn on
            }


            if (newMeasurement.ReferenceIntegrationTime == -1)
            {
            	//UI.Utils.DrawProgressBar("Processing", "Setting reference integration time...", newMeasurement.Lot.ToUpper(), 0.15f); 
            	DrawProgressRect(pbX, pbY, pbWidth, pbHeight, 0.15f); 

                newMeasurement.ReferenceIntegrationTime = DetectIntegrationTime(0.8f, 10, 0, 1);
                if (newMeasurement.ReferenceIntegrationTime <= 0)
                {
                    Lamp.Off();
                    Shutter.Close();
                    GenericMessage.DrawMultiline(this.Name,
                                        "Error setting integration time!",
                                        new string[] {"Ensure the sample being measured is stable and try", 
                                                                "measuring again."}, false);
                     
                    Beeper.On();
                    Thread.Sleep(1000);
                    Beeper.Off();
                    Shutter.Stop();
                    return false;
                }
            }

            //UI.Utils.DrawProgressBar("Processing", "Recording reference spectra...", newMeasurement.Lot.ToUpper(), 0.35f); 
            DrawProgressRect(pbX, pbY, pbWidth, pbHeight, 0.35f); 
            
            newMeasurement.LightReferenceSpectra = new float[256];
            Spectrometer.Read(newMeasurement.ReferenceIntegrationTime, newMeasurement.ReferenceScansToAverage);
            Array.Copy(Spectrometer.LastSpectra, newMeasurement.LightReferenceSpectra, 256);


            Core.Hardware.Shutter.Open();
             
            Thread.Sleep(250);
            Shutter.Stop();

            if (newMeasurement.SampleIntegrationTime == -1)
            {
            	//UI.Utils.DrawProgressBar("Processing", "Setting specimen integration time...", newMeasurement.Lot.ToUpper(), 0.45f); 
            	DrawProgressRect(pbX, pbY, pbWidth, pbHeight, 0.45f); 
            	
                newMeasurement.SampleIntegrationTime = DetectIntegrationTime(0.8f, 10, 0, 1);
                if (newMeasurement.SampleIntegrationTime <= 0)
                {
                    Lamp.Off();
                    Shutter.Close();
                    GenericMessage.DrawMultiline(this.Name,
                                        "Error setting integration time!",
                                        new string[] {"Ensure the sample being measured is stable and try", 
                                                                "measuring again."}, false);
                    Beeper.On();
                    Thread.Sleep(1000);
                    Beeper.Off();
                    Shutter.Stop(); 
                    return false;
                }
            }

            //UI.Utils.DrawProgressBar("Processing", "Recording specimen spectra...", newMeasurement.Lot.ToUpper(), 0.55f); 
            DrawProgressRect(pbX, pbY, pbWidth, pbHeight, 0.55f); 
            
          
            newMeasurement.LightSpecimenSpectra = new float[256];
            Spectrometer.Read(newMeasurement.SampleIntegrationTime, newMeasurement.SpecimenScansToAverage);
            Array.Copy(Spectrometer.LastSpectra, newMeasurement.LightSpecimenSpectra, 256);
 
            Core.Hardware.Lamp.Off(); 

            newMeasurement.DarkSpecimenSpectra = new float[256];
            if (Options.EstimateSpecimenDarkSpectra)
            {
                float minValue = MathUtils.Min(newMeasurement.LightSpecimenSpectra);
                for (int i = 0; i < newMeasurement.DarkSpecimenSpectra.Length; i += 1)
                {
                    newMeasurement.DarkSpecimenSpectra[i] = minValue;
                }
            }
            else
            {
                Thread.Sleep(1000); //Wait for the light to turn off
                Spectrometer.Read(newMeasurement.SampleIntegrationTime, newMeasurement.SpecimenScansToAverage);
                Array.Copy(Spectrometer.LastSpectra, newMeasurement.DarkSpecimenSpectra, 256);
            }
             
            Core.Hardware.Shutter.Close();
 
            newMeasurement.DarkReferenceSpectra = new float[256];
            if (Options.EstimateReferenceDarkSpectra)
            {
                float minValue = MathUtils.Min(newMeasurement.LightReferenceSpectra);
                for (int i = 0; i < newMeasurement.DarkReferenceSpectra.Length; i += 1)
                {
                    newMeasurement.DarkReferenceSpectra[i] = minValue;
                }
            }
            else
            {
		Thread.Sleep(250);
		Shutter.Stop();
                Spectrometer.Read(newMeasurement.ReferenceIntegrationTime, newMeasurement.ReferenceScansToAverage);
                Array.Copy(Spectrometer.LastSpectra, newMeasurement.DarkReferenceSpectra, 256);
            }

            if (GPS.Enabled)
            {
                //UI.Utils.DrawProgressBar("Processing", "Recording GPS location...", newMeasurement.Lot.ToUpper(), 0.65f);
                DrawProgressRect(pbX, pbY, pbWidth, pbHeight, 0.65f); 
                 
                Core.Hardware.GPS.UpdateStatus(true);

                bool skipGPS = false;
                 
                AddressSpace.Write((uint)SRAM.MemoryMap.BootupTemporaryScreenBuffer, Display.Buffer, 0, Display.Buffer.Length);

                while ((!Core.Hardware.GPS.Fix) && (!skipGPS))
                {
                    Environment.KeyPad.keyCode = 0;
                    ErrorMessage.Draw(true, "GPS Error", ErrorMessage.MessageType.ContinueRetry);

                    while (Environment.KeyPad.keyCode == 0)
                    {
                        Environment.KeyPad.SetKeyCode();
                        Watchdog.Reset();
                        Thread.Sleep(100); 
                    }

                    if (Environment.KeyPad.keyCode == KeyPad.KeyCode.Left)
                    {
                        if (Settings.Read("Device.Keypad.Sounds", true))
                        {
                            Beeper.PlayKeypadSound();
                        }
                        GenericMessage.DrawCentered("Reading GPS...");
                        Core.Hardware.GPS.UpdateStatus(false);

                        if (Core.Hardware.GPS.Fix)
                        {
                            AddressSpace.Read((uint)SRAM.MemoryMap.BootupTemporaryScreenBuffer, Display.Buffer, 0, Display.Buffer.Length);
                            Display.Update(Display.Buffer);
                            break;
                        }
                    }
                    else if (Environment.KeyPad.keyCode == KeyPad.KeyCode.Right)
                    {
                        if (Settings.Read("Device.Keypad.Sounds", true))
                        {
                            Beeper.PlayKeypadSound();
                        }
                        GenericMessage.DrawCentered("Ignoring GPS failure...");
                        skipGPS = true;
                        AddressSpace.Read((uint)SRAM.MemoryMap.BootupTemporaryScreenBuffer, Display.Buffer, 0, Display.Buffer.Length);
                        Display.Update(Display.Buffer);
                        break;
                    }

                }
                Environment.KeyPad.keyCode = 0;

                if (Core.Hardware.GPS.Fix)
                {
                    newMeasurement.Longitude = Core.Hardware.GPS.Longitude;
                    newMeasurement.Latitude = Core.Hardware.GPS.Latitude;
                    newMeasurement.GPSFixQuality = Core.Hardware.GPS.FixQuality;
                    newMeasurement.GPSSatellites = (byte)Core.Hardware.GPS.Satellites;
                }
                else
                {
                    newMeasurement.Longitude = Core.Hardware.GPS.LastLongitude;
                    newMeasurement.Latitude = Core.Hardware.GPS.LastLatitude;
                    newMeasurement.GPSFixQuality = 0;
                    newMeasurement.GPSSatellites = 0;
                }
            }

            newMeasurement.WavelengthInterpolationC0 = SpectraUtils.C0;
            newMeasurement.WavelengthInterpolationC1 = SpectraUtils.C1;
            newMeasurement.WavelengthInterpolationC2 = SpectraUtils.C2;
            newMeasurement.WavelengthInterpolationC3 = SpectraUtils.C3;
            newMeasurement.WavelengthInterpolationC4 = SpectraUtils.C4;
            newMeasurement.WavelengthInterpolationC5 = SpectraUtils.C5;

            newMeasurement.Models = this.Models;

            base.BuildSpectra(newMeasurement);

            //UI.Utils.DrawProgressBar("Processing", "Saving measurement...", newMeasurement.Lot.ToUpper(), 0.95f);
	    DrawProgressRect(pbX, pbY, pbWidth, pbHeight, 1f); 
	    
	    Shutter.Stop();

            return true;
        }

        public virtual void SetPredictions(AppMeasurement aMeasurement)
        {
            base.SetPredictions(aMeasurement);
        }
         

        public override void ShowMeasurementMenu()
        {
            string menuSelection = "";
            string defaultMenuSelection = "Delete Measurement";

            ExitAppMenu = false;
            while (!Environment.ShutingDown && !ExitAppMenu)
            {
                menuSelection = UI.Utils.ShowMenu(new string[] { "Delete Measurement", "Move Measurement", "Delete Lot" }, defaultMenuSelection);
                defaultMenuSelection = menuSelection;

                if (UI.Utils.ExitMenu)
                {
                    ExitAppMenu = true;
                    break;
                }

                if (menuSelection == "<-")
                {
                    UI.Utils.ShowMeasurement(Environment.CurrentFile);
                    return;
                }
                else if (menuSelection == "Move Measurement")
                {
                    string lotMenuSelection = "";

                    while (lotMenuSelection != "<-")
                    {
                        if (Environment.RecentLots.Length > 0)
                        {
                            lotMenuSelection = UI.Utils.ShowMenu(new string[] { "Select Recently Used Lots", "Select Lot from List", "Edit Lot Manually" }, defaultMenuSelection);
                        }
                        else
                        {
                            lotMenuSelection = UI.Utils.ShowMenu(new string[] { "Select Lot from List", "Edit Lot Manually" }, defaultMenuSelection);
                        }

                        if (lotMenuSelection == "Select Recently Used Lots")
                        {
                            if (Environment.RecentLots.Length > 0)
                            {
                                lotMenuSelection = UI.Utils.ShowMenu(Environment.RecentLots, Environment.RecentLots[0]);

                                if (lotMenuSelection != "<-")
                                {
                                    bool success = LotUtils.MoveMeasurement(Environment.CurrentFile, lotMenuSelection, Options.AllowOverwrite);
                                    if (success)
                                    {
                                        UI.Utils.ShowMeasurement(Environment.CurrentFile);
                                        return;
                                    }
                                    else
                                    {
                                        GenericMessage.DrawMultiline("Error Moving Measurement",
                                                            "Cannot Overwrite Existing Measurement",
                                                            new string[] {"The measurement could not be moved because it would", 
                                                                "overwrite another measurement and the model option",
                                                                "[AllowOverwrite] is set to [No]"}, true);
                                    }
                                }
                            }
                        }
                        else if (lotMenuSelection == "Select Lot from List")
                        {

                            if (!System.IO.File.Exists(LotUtils.Root + "LotsList.txt"))
                            {
                                GenericMessage.DrawMultiline("No list on SD card!",
                                                           "Lots List File Does Not Exist",
                                                           new string[] {"Create a file called LotsList.txt with one line per",
                                                                "lot to display on the SD card under the folder",
                                                                LotUtils.Root}, true);
                            }
                            else
                            {
                                string[] lines = FileUtils.ReadAllLines(LotUtils.Root + "LotsList.txt");

                                if (lines.Length > 0)
                                {
                                    lotMenuSelection = UI.Utils.ShowMenu(lines, lines[0]);

                                    if (lotMenuSelection != "<-")
                                    {
                                        bool success = LotUtils.MoveMeasurement(Environment.CurrentFile, lotMenuSelection, Options.AllowOverwrite);
                                        if (success)
                                        {
                                            UI.Utils.ShowMeasurement(Environment.CurrentFile);
                                            return;
                                        }
                                        else
                                        {
                                            GenericMessage.DrawMultiline("Error Moving Measurement",
                                                                "Cannot Overwrite Existing Measurement",
                                                                new string[] {"The measurement could not be moved because it would", 
                                                                "overwrite another measurement and the model option",
                                                                "[AllowOverwrite] is set to [No]"}, true);
                                        }
                                    }
                                }
                            }


                        }
                        else if (lotMenuSelection == "Edit Lot Manually")
                        {
                            string measurementLot = LotUtils.GetRelativeLotPath(System.IO.Path.GetDirectoryName(Environment.CurrentFile));
                            if (measurementLot.Length == 0)
                            {
                                measurementLot = "\\ ";
                            }

                            string textInput = UI.Utils.TextInput(UI.Utils.TEXT_ALPHABET + "\\", "Enter the lot for this measurement", measurementLot);

                            if (textInput != Environment.CurrentLot)
                            {
                                bool success = LotUtils.MoveMeasurement(Environment.CurrentFile, textInput, Options.AllowOverwrite);
                                if (success)
                                {
                                    UI.Utils.ShowMeasurement(Environment.CurrentFile);
                                    return;
                                }
                                else
                                {
                                    GenericMessage.DrawMultiline("Error Moving Measurement",
                                                        "Measurement could not be moved",
                                                        new string[] {"Lot elements must be seperated by \\ and must contain", 
                                                                "an incrementing portion padded with zeros.  E.g. lot path...",
                                                                "\\GRP001\\LOT001\\SMP001"}, true);
                                }
                            }
                        }
                    }
                }
                else if (menuSelection == "Delete Measurement")
                {
                    Core.Screens.GenericMessage.DrawCentered("Deleting measurement...");

                    DeleteMeasurement(Environment.CurrentFile);
                    string currentFolder = LotUtils.GetFullLotPath(Environment.CurrentLot);
                    Environment.CurrentFile = "";

                    while (true)
                    {
                        if (
                            (!System.IO.Directory.Exists(currentFolder)) ||
                            (FileUtils.CountFiles(currentFolder, ".f750dat", true) == 0)
                            )
                        {
                            currentFolder = LotUtils.GetParentLot(currentFolder);

                            if (LotUtils.IsRootFolder(currentFolder))
                            {
                                UI.Utils.ShowMainMenu(2);
                                break;
                            }
                        }
                        else
                        {
                            Environment.CurrentLot = currentFolder;
                            UI.Utils.ShowMeasurement(Environment.CurrentLot);
                            break;
                        }

                    }

                    return;
                }
                else if (menuSelection == "Delete Lot")
                {
                    Core.Screens.GenericMessage.DrawCentered("Deleting lot...");
                    string fullPath = LotUtils.GetFullLotPath(Environment.CurrentLot); 
                    DirectoryInfo dirInfo = new System.IO.DirectoryInfo(fullPath);

                    string[] files = System.IO.Directory.GetFiles(fullPath);
                    foreach (string file in files)
                    {
                        if (System.IO.Path.GetExtension(file).ToLower().Equals(".f750dat"))
                        {
                            DeleteMeasurement(file);
                        }
                    }

                    dirInfo = new System.IO.DirectoryInfo(dirInfo.Parent.FullName);
                    string currentFolder = dirInfo.FullName;
                    while (true)
                    {
                        if (LotUtils.IsRootFolder(currentFolder))
                        {
                            Environment.CurrentLot = LotUtils.GetRelativeLotPath(currentFolder);
                            break;
                        }

                        LotUtils.DeleteEmptyLots(currentFolder);
                        if (System.IO.Directory.Exists(currentFolder)) //If the directory still exists it has files, so use this folder.
                        {
                            Environment.CurrentLot = LotUtils.GetRelativeLotPath(currentFolder);
                            break;
                        }

                        dirInfo = new System.IO.DirectoryInfo(dirInfo.Parent.FullName);
                        currentFolder = dirInfo.FullName;
                    }

                    if (System.IO.Directory.GetDirectories(currentFolder).Length == 0)
                    {
                        Environment.CurrentLot = "";
                        UI.Utils.ShowMainMenu(2);
                    }
                    else
                    {
                        Environment.CurrentLot = currentFolder;
                        UI.Utils.ShowMeasurement(Environment.CurrentLot);
                    }
                       
                    return;
                }  

            }

        }

        public bool InitUploadProcess()
        {  
            string networkSSID = Core.Hardware.Esp32.SSID;
            string networkPassword = Core.Hardware.Esp32.NetworkKey;
             
            int retryCount = 0;

            try
            {
                UI.Utils.DrawProgressBar("Processing", "Connecting to network...", "SSID: " + networkSSID, 0.07f);
                Core.Hardware.Esp32.Init();

                UI.Utils.DrawProgressBar("Processing", "Connecting to network...", "SSID: " + networkSSID, 0.10f);
                Core.Hardware.Esp32.Connect(networkSSID, networkPassword);
                retryCount = 0;
                while ((!Core.Hardware.Esp32.IsConnected(networkSSID)) &&
                        (retryCount < 50))
                {
                    retryCount += 1;
                }
                if (retryCount >= 3)
                { 
                    return false;
                }
                 
            }
            catch 
            { 
                return false;
            }

            return true;
        }

        public bool ConnectToDataServer()
        { 
            int retryCount = 0;

            try
            { 
                UI.Utils.DrawProgressBar("Processing", "Connecting to data server", "tcp://" + Options.DataServer + ":" + Options.DataServerPort, 0.15f);
                retryCount = 0;
                while ((!Core.Hardware.Esp32.ConnectServer(Options.DataServer, Options.DataServerPort, false, 7200)) &&
                        (retryCount < 3))
                {
                    retryCount += 1;
                }
                if (retryCount >= 3)
                {
                    return false;
                }
            }
            catch  
            {
                return false;
            }

            return true;
        }


        public void DeinitUploadProcess()
        {
            UI.Utils.DrawProgressBar("Processing", "Turning Wifi off", "", 1f); 
            Core.Hardware.Esp32.Deinit(); 
        }







          
        public int MeasurementsToUpload = 0;
        public int MeasurementsUploaded = 0; 

        public bool UploadMeasurements(string aPath)
        { 
             
            if (System.IO.Directory.Exists(aPath))
            {
                string[] files = System.IO.Directory.GetFiles(aPath); 

                for (int i = 0; i < files.Length; i += 1)
                {
                    if (files[i].IndexOf(".f750dat") > -1)
                    {
                        if (!UploadMeasurement(files[i]))
                        {
                            return false;
                        }
                    }
                }

                if (System.IO.Directory.Exists(aPath))
                {
                    string[] folders = System.IO.Directory.GetDirectories(aPath);

                    for (int i = 0; i < folders.Length; i += 1)
                    {
                        if (!UploadMeasurements(folders[i]))
                        {
                            return false;
                        }
                    }
                }
            }

            return true;
        }
          
        public bool UploadTextFile(string aTextFile, float aPercProgress)
        {
            if (!System.IO.File.Exists(aTextFile))
            {
                return false;
            }

            int retryCount = 0;

            

            string textFilename = System.IO.Path.GetFileName(aTextFile);

            UI.Utils.DrawProgressBar("Processing", "Uploading to Data Server", "Text measurement log file...", aPercProgress);
             
            StringBuilder dataString = new StringBuilder();

            FileStream fs = System.IO.File.OpenRead(aTextFile);
            fs.Seek(0, SeekOrigin.Begin);

            string textFileNameBytes = StringUtils.ByteArrayToHexViaLookup(Encoding.UTF8.GetBytes(textFilename));
            
            if (Options.DataServer != "")
            { 
                retryCount = 0;
                while ((Core.Hardware.Esp32.SendServerData(Options.DataServer, Options.DataServerPort, "GET /F75X/ HTTP/1.1\r\n" +
                        "Accept: */*\r\n" +
                        "Host: " + Options.DataServer + "\r\n" +
                        "User-Agent: F75X" +
                        "Connection: keep-alive\r\n\r\n\n").Length == 0) && (retryCount < 3))
                {
                    retryCount += 1;
                }
                if (retryCount >= 3)
                {
                    Core.Screens.GenericMessage.DrawCentered("Data server error.", true);
                    return false;
                }

                byte[] commandBuffer = new byte[0];
                string stringBuffer = "";

                while (fs.Position != fs.Length)
                {
                    if (commandBuffer.Length != System.Math.Min(800, fs.Length - fs.Position))
                    {
                        commandBuffer = new byte[(int)System.Math.Min(800, fs.Length - fs.Position)];
                    }

                    fs.Read(commandBuffer, 0, (int)commandBuffer.Length);

                    stringBuffer = StringUtils.ByteArrayToHexViaLookup(commandBuffer);

                    retryCount = 0;
                    while ((Core.Hardware.Esp32.SendServerData(Options.DataServer, Options.DataServerPort,
                        "GET " +
                        "/F75X/?device=" + Core.Hardware.Device.SerialNumber.Replace(" ","") + "&log=" + textFileNameBytes + "&len=" + fs.Length + "&size=" + stringBuffer.Length + "&bin=" + stringBuffer +
                        " HTTP/1.1\r\n" +
                        "Accept: */*\r\n" +
                        "Host: " + Options.DataServer + "\r\n" +
                        "User-Agent: F75X" +
                        "Connection: keep-alive\r\n\r\n\n").Length == 0) && (retryCount < 3))
                    {
                        retryCount += 1;
                    }
                    if (retryCount >= 3)
                    {
                        break;
                    }

                }
            }

            fs.Close();
            fs = null;

            if (retryCount >= 3)
            {
                return false;
            }
            else
            {
                return true;
            }

        }
 
        public bool UploadMeasurement(string aMeasurementFile)
        {
            int retryCount = 0;

            float progress = 0.20f + (0.6f * ((float)(1 + MeasurementsUploaded) / (float)MeasurementsToUpload));

            UI.Utils.DrawProgressBar("Processing", "Uploading to Data Server", "Measurement " + (1 + MeasurementsUploaded) + " of " + MeasurementsToUpload + "...", progress);

            StringBuilder dataString = new StringBuilder();

            FileStream fs = System.IO.File.OpenRead(aMeasurementFile);

            Measurement measurementToUpload = new Measurement();
            measurementToUpload.FromStream(fs);
            string measurementGuid = StringUtils.ByteArrayToHexViaLookup(measurementToUpload.MeasurementID.ToByteArray());
            fs.Position = 0;

            retryCount = 0;
            while ((Core.Hardware.Esp32.SendServerData(Options.DataServer, Options.DataServerPort, "GET /F75X/ HTTP/1.1\r\n" +
                    "Accept: */*\r\n" +
                    "Host: " + Options.DataServer + "\r\n" +
                    "User-Agent: F75X" +
                    "Connection: keep-alive\r\n\r\n\n").Length == 0) && (retryCount < 3))
            {
                retryCount += 1;
            }
            if (retryCount >= 3)
            {
                fs.Close();
                return false; //Error 
            }

            byte[] commandBuffer = new byte[0];
            string stringBuffer = "";

            while (fs.Position != fs.Length)
            {
                int index = (int)fs.Position;

                if (commandBuffer.Length != System.Math.Min(900, fs.Length - fs.Position))
                {
                    commandBuffer = new byte[(int)System.Math.Min(900, fs.Length - fs.Position)];
                }

                fs.Read(commandBuffer, 0, (int)commandBuffer.Length);

                stringBuffer = StringUtils.ByteArrayToHexViaLookup(commandBuffer);

                string serverResponse = Core.Hardware.Esp32.SendServerData(Options.DataServer, Options.DataServerPort,
                    "GET " +
                    "/F75X/?device=" + Core.Hardware.Device.SerialNumber.Replace(" ", "") + "&dat=" + measurementGuid + "&len=" + fs.Length + "&index=" + index + "&size=" + stringBuffer.Length + "&bin=" + stringBuffer +
                    " HTTP/1.1\r\n" +
                    "Accept: */*\r\n" +
                    "Host: " + Options.DataServer + "\r\n" +
                    "User-Agent: F75X" +
                    "Connection: keep-alive\r\n\r\n\n");

                if ((serverResponse.Length > 1) && serverResponse.Substring(serverResponse.Length - 1, 1).Equals("0"))
                {
                    fs.Close();
                    MeasurementsUploaded += 1; 
                    return true; //Measurement existed and was skipped
                }

            }
            MeasurementsUploaded += 1; 

            fs.Close();
            fs = null;

            return true; //Measurement uploaded
        }
 

         

        public string[] AppMenuOptions = new string[] { "Current Bin", "Default Tags", "Archive Measurements", "Advanced Options", "Notes" };

        public override void ShowAppMenu()
        {
            string menuSelection = "";
            string defaultMenuSelection = "Show Preview";

            ExitAppMenu = false;
            while (!Environment.ShutingDown && !ExitAppMenu)
            {
                menuSelection = UI.Utils.ShowMenu(AppMenuOptions, defaultMenuSelection);
                defaultMenuSelection = menuSelection;

                if (UI.Utils.ExitMenu)
                {
                    ExitAppMenu = true;
                    break;
                }

                if (menuSelection == "<-")
                {
                    UI.Utils.ShowMainMenu(1);
                    break;
                }
                else if (menuSelection == "Current Bin")
                {
			string measurementLot = LotUtils.GetIncrementingPortion(Options.DefaultLot, "BIN");
			  
			string textInput = "";

			while (true)
			{
				textInput = UI.Utils.TextInput("0123456789", "Enter BIN # for new measurements.\n", measurementLot);

				if (textInput.Trim().TrimEnd('\\').Length == 0)
				{
					GenericMessage.DrawCentered("Error: No bin specified.", true);
				}
				else
				{
					break;
				}
			}
 
                        if (textInput != "")
                        {
				Options.DefaultLot = "\\BIN" + StringUtils.ZeroPadString(int.Parse(textInput), 3) + "\\FRUIT001\\SIDE001";

				    //Options.IncrementationLogic_Lots     = new string[] { "BIN", "FRUIT", "SIDE" };
				    //Options.IncrementationLogic_Maximums = new int[] { 10, 2, 3 };

				SaveOptions();
                        }
                }
                else if (menuSelection == "Default Tags")
                {
                    string textInput = UI.Utils.TextInput("Enter tags for new measurements\n" +
                                                          "e.g. Fruit, Farm, User, Project", Options.DefaultTags);

                    if (textInput != Options.DefaultTags)
                    {
                        Options.DefaultTags = textInput;
                    }
                }
                else if (menuSelection.Equals("Lot Incrementation Logic"))
                {  
                    string lotToChangeLogicFor = Options.IncrementationLogic_Lots[0];

                    while (lotToChangeLogicFor != "<-")
                    {
                        lotToChangeLogicFor = UI.Utils.ShowMenu(Options.IncrementationLogic_Lots, lotToChangeLogicFor);

                        int pos = Array.IndexOf(Options.IncrementationLogic_Lots, lotToChangeLogicFor);

                        if (pos != -1)
                        {
                            int defaultValue = defaultValue = Options.IncrementationLogic_Maximums[pos];

                            int measurementsPerLot = UI.Utils.IntegerInput("Enter the count that triggers\nincrementing this lot.", defaultValue, 1, 999);

                            if (measurementsPerLot != int.MinValue)
                            {
                                Options.IncrementationLogic_Maximums[pos] = measurementsPerLot;
                                SaveOptions();
                            }
                        }
                    }

                }
                else if (menuSelection == "Archive Measurements")
                {
                    int measurementCount = 0;

                    if (System.IO.Directory.Exists(LotUtils.Root))
                    {
                        measurementCount = LotUtils.GetTotalMeasurementCount();
                    }

                    bool continueArchving = true;
                    string archiveFolderPath = "";
                    string folderName = DateTime.Now.ToString("yyyy_MM_dd_HH_mm_ss");

                    if (measurementCount == 0)
                    {
                        Core.Screens.GenericMessage.DrawCentered("No measurements to archive.", true);
                        continueArchving = false;
                    }

                    if (continueArchving)
                    { 
                        UI.Utils.DrawProgressBar("Processing", "Preparing SD Card...", "Creating folder...", 0.05f);
             
                        if (!System.IO.Directory.Exists(Environment.ArchiveFolder))
                        {
                            System.IO.Directory.CreateDirectory(Environment.ArchiveFolder);
                        }

                        archiveFolderPath = Environment.ArchiveFolder + folderName + "\\";                     
                    }
                     

                    if (continueArchving)
                    {
                        if ((Esp32.Enabled) && (Options.DataServer.Trim().Length != 0) && (!Core.Hardware.Esp32.SSID.Equals("NotSet")))
                        {
                            if (InitUploadProcess())
                            {
                                if (ConnectToDataServer())
                                {
                                    UploadTextFile(@"\SD\" + TextOutputFile, 0.15f);

                                    MeasurementsUploaded = 0;
                                    MeasurementsToUpload = measurementCount;
                                    if (!UploadMeasurements(LotUtils.Root))
                                    {
                                        GenericMessage.DrawMultiline("Data Server Upload Error",
                                                            (measurementCount - MeasurementsUploaded) + " of " + measurementCount + " failed to upload.",
                                                            new string[] { "This may be due to low signal quality or network traffic.", 
                                                                        "Retry or bypass uploading by disabling wireless networking", 
                                                                        "under Main Menu\\Device Setup\\Wireless Network." }, true);
                                        continueArchving = false;
                                    }

                                }
                                else
                                {
                                    GenericMessage.DrawMultiline("Data Server Upload Error",
                                                               "tcp://" + Options.DataServer + ":" + Options.DataServerPort,
                                                               new string[] { "Verify compatible Felix Data Server software is running on",
                                                                        "the computer at this address and try again.  Or bypass",
                                                                        "data uploading by disabling the wireless networking feature",
                                                                        "under Main Menu\\Device Setup\\Wireless Network." }, true);
                                    continueArchving = false;
                                }
                                DeinitUploadProcess(); 
                            }
                            else
                            {
                                GenericMessage.DrawMultiline("Data Server Upload Error",
                                                    "Network SSID: " + Esp32.SSID,
                                                    new string[] { "Count not connect to network.  Reconfigure the connection",
                                                                        "or disable wireless networking to by pass this feature",
                                                                         "under Main Menu\\Device Setup\\Wireless Network"}, true); 
                                continueArchving = false;
                            }
                        }
                    }


                    if (continueArchving)
                    {
                        UI.Utils.DrawProgressBar("Processing", "Moving measurements...", "Folder: " + "\\Archive\\" + folderName, 0.9f);
             
                        try
                        {
                            System.IO.Directory.Move(LotUtils.Root, archiveFolderPath);
                            System.IO.File.Move(@"\SD\" + TextOutputFile, archiveFolderPath + TextOutputFile);
                        }
                        catch (Exception ex)
                        {
                            Microsoft.SPOT.Debug.Print(ex.ToString());
                        }

                        GenericMessage.DrawMultiline("Archive Measurements",
                                            measurementCount + " measurements processed.",
                                            new string[] { "Text log and measurements stored in folder...",
                                                            "\\Archive\\" + folderName}, true); 
                    }
                     
                }
                else if (menuSelection == "Advanced Options")
                { 
                    string[] advancedMenuItems = null;

                    if (Models.Length > 0) 
                    {
                        advancedMenuItems = new string[] { "Sample Scans to Average", "Reference Scans to Average", "Log File", "Data Server", "Bias Adjustments", "Load Default Options" };
                    } 
                    else 
                    {
                        advancedMenuItems = new string[] { "Sample Scans to Average", "Reference Scans to Average", "Log File", "Data Server", "Load Default Options" };
                    }

                    string optionsSelection = "Sample Scans to Average";
                    while (optionsSelection != "<-")
                    {
                        optionsSelection = UI.Utils.ShowMenu(advancedMenuItems, optionsSelection);

                        if (optionsSelection.Equals("Bias Adjustments"))
                        {
                            string calibrationAdjustmentsMenu = "";
                            for (int i = 0; i < Models.Length; i += 1)
                            {
                                calibrationAdjustmentsMenu += Models[i].Name;
                                if (i < Models.Length -1)
                                {
                                    calibrationAdjustmentsMenu += ",";
                                }
                            }
                            string calibrationAdjustmentsMenuSelection = Models[0].Name;
                            while (calibrationAdjustmentsMenuSelection != "<-")
                            {
                                calibrationAdjustmentsMenuSelection = UI.Utils.ShowMenu(calibrationAdjustmentsMenu.Split(','), calibrationAdjustmentsMenuSelection);

                                if (calibrationAdjustmentsMenuSelection != "<-")
                                {
                                    int selectedModel = Array.IndexOf(calibrationAdjustmentsMenu.Split(','), calibrationAdjustmentsMenuSelection);

                                    string biasOrInteceptMenuSelection = "Intercept";
                                    while (biasOrInteceptMenuSelection != "<-")
                                    {
                                        biasOrInteceptMenuSelection = UI.Utils.ShowMenu(new string[] { "Intercept", "Slope" }, biasOrInteceptMenuSelection);
                                        if (biasOrInteceptMenuSelection == "Intercept")
                                        {
                                            float adj = UI.Utils.DecimalInput("Enter the bias correction intercept for\n" + Models[selectedModel].Name, Models[selectedModel].BiasCorrectionIntercept, -9999, 9999, 4);

                                            if (adj != float.MinValue)
                                            {
                                                Models[selectedModel].BiasCorrectionIntercept = adj;
                                                SaveOptions(); 
                                            }
                                        }
                                        else if (biasOrInteceptMenuSelection == "Slope")
                                        {
                                            float adj = UI.Utils.DecimalInput("Enter the bias correction slope for\n" + Models[selectedModel].Name, Models[selectedModel].BiasCorrectionSlope, -9999, 9999, 4);

                                            if (adj != float.MinValue)
                                            {
                                                Models[selectedModel].BiasCorrectionSlope = adj;
                                                SaveOptions(); 
                                            }
                                        } 
                                    }
                                }
                            }
                        }
                        else if (optionsSelection.Equals("Sample Scans to Average"))
                        {
                            int scansToAvg = UI.Utils.IntegerInput("Enter the number of scans to average to use for\n" +
                                                              "sample measurements.", Options.SampleScansToAverage, 1, 128);

                            if (scansToAvg != int.MinValue)
                            {
                                Options.SampleScansToAverage = scansToAvg;
                                SaveOptions();
                            }
                        }
                        else if (optionsSelection.Equals("Reference Scans to Average"))
                        {
                            int scansToAvg = UI.Utils.IntegerInput("Enter the number of scans to average to use for\n" +
                                                              "reference measurements.", Options.ReferenceScansToAverage, 1, 128);

                            if (scansToAvg != int.MinValue)
                            {
                                Options.ReferenceScansToAverage = scansToAvg;
                                SaveOptions();
                            }
                        }
                        else if (optionsSelection.Equals("Log File"))
                        {
                            TextOutputFile = UI.Utils.TextInput("Enter the file to log measurements to\n" +
                                                                "or leave this blank disable logging.", TextOutputFile).Trim();
                            SaveOptions();
                        } 
                        else if (optionsSelection.Equals("Load Default Options"))
                        {
                            Core.Screens.GenericMessage.DrawCentered("Loading default options...", false);
                            int itemsToRemove = 0;

                            for (int i = 0; i < Settings.Keys.Length; i += 1)
                            {
                                if (Settings.Keys[i].IndexOf(this.Name) == 0)
                                {
                                    itemsToRemove += 1;
                                }
                            }

                            string[] newKeys = new string[Settings.Keys.Length - itemsToRemove];
                            string[] newValues = new string[Settings.Keys.Length - itemsToRemove];

                            int n = 0;
                            for (int i = 0; i < Settings.Keys.Length; i += 1)
                            {
                                if (Settings.Keys[i].IndexOf(this.Name) != 0)
                                {
                                    newKeys[n] = Settings.Keys[i];
                                    newValues[n] = Settings.Values[i];
                                    n += 1;
                                }
                            }

                            Settings.Keys = newKeys;
                            Settings.Values = newValues;
                            Settings.Save();
                            if (System.IO.Directory.Exists(Environment.SettingsFolder))
                            {
                                string[] files = System.IO.Directory.GetFiles(Environment.SettingsFolder);
                                foreach (string file in files)
                                {
                                    if ((file.ToLower().IndexOf(this.ID.ToString()) != 0)
                                            &&
                                         (System.IO.Path.GetExtension(file).ToLower().Equals(".txt")))
                                    {
                                        System.IO.File.Delete(file);
                                    }
                                } 
                            }
                            Init();
                            LoadOptions();
                            SaveOptions(); 
                        }
                        else if (optionsSelection.Equals("Data Server"))
                        {
                            string defaultServerValue = "";

                            if (Options.DataServer.Trim().Length != 0)
                            {
                                if (Options.DataServerPort.Equals("80"))
                                {
                                    defaultServerValue = Options.DataServer;
                                }
                                else
                                {
                                    defaultServerValue = Options.DataServer + ":" + Options.DataServerPort;
                                }
                            }
                            else
                            {
                                defaultServerValue = " ";
                            }

                            string textInput = UI.Utils.TextInput("ABCDEFGHIJKLMNOPQRSTUVWXYZ.:0123456789 ",
                                                                    "Enter the address of the data server\n" +
                                                                    "used to archive measurements.", defaultServerValue);

                            if (textInput.Trim().Length == 0)
                            {
                                Options.DataServer = "";
                                Options.DataServerPort = "";
                            }
                            else
                            {
                                if (textInput.IndexOf(":") > -1)
                                {
                                    Options.DataServer = textInput.Split(':')[0];
                                    Options.DataServerPort = textInput.Split(':')[1];
                                }
                                else
                                {
                                    Options.DataServer = textInput;
                                    Options.DataServerPort = "80";
                                }
                                SaveOptions();
                            }

                        }  
                    }
                }
                else if (menuSelection == "Notes")
                {
                    ShowNotes();
                    Environment.KeyPad.WaitForKeypress();
                }
            }


        }
 
        
	public void DeleteLot(string aFullLotPath)
        {  
		string[] folders = System.IO.Directory.GetDirectories(aFullLotPath); 

        	foreach (string folder in folders)
		{  
			DeleteLot(folder);
		}
		   
                if (System.IO.Directory.Exists(aFullLotPath))
                {
			string[] files = System.IO.Directory.GetFiles(aFullLotPath);  

			if (files != null)
			{
				foreach (string file in files)
				{   
					if (file.ToLower().IndexOf(".f750dat") > -1)
					{ 
						DeleteMeasurement(file); 
					}
				}   
			}
		} 
        }
        
        public override void DeleteMeasurement(string aMeasurementFile)
        {   
	    AppMeasurement measurement = new AppMeasurement();  
	    measurement.FromBytes(System.IO.File.ReadAllBytes(aMeasurementFile)); 

	    string path = System.IO.Path.GetDirectoryName(aMeasurementFile); 


	    LotUtils.RemoveStats(path, 0, 0f); 
	    for (int i = 0; i < measurement.PredictionValues.Length; i += 1)
	    {
		LotUtils.RemoveStats(path, i + 1, measurement.PredictionValues[i]);  
	    }

	   // base.RemoveFromLog(measurement);

	    base.DeleteMeasurement(aMeasurementFile);   
	    LotUtils.DeleteEmptyLots(path);    
        }

        public override void SaveMeasurement(Measurement aMeasurement)
        {
            //GenericMessage.DrawCentered("1.1", true);

            AppMeasurement newMeasurement = (AppMeasurement)aMeasurement;
        
            newMeasurement.Filename = LotUtils.GetNextMeasurementFilename(newMeasurement.Lot);
               
            //GenericMessage.DrawCentered("1.2", true); 
            
            Environment.CurrentFile = newMeasurement.Filename; 
            Environment.CurrentLot = newMeasurement.Lot;  

            LotUtils.AddStats(newMeasurement.Lot, 0, 0f);

            //GenericMessage.DrawCentered("1.2", true);
            for (int i = 0; i < newMeasurement.PredictionValues.Length; i += 1)
            {
                LotUtils.AddStats(newMeasurement.Lot, i + 1, newMeasurement.PredictionValues[i]);
            }
            //GenericMessage.DrawCentered("1.3", true);

            if (TextOutputFile != "")
            { 
                System.IO.FileStream fs = System.IO.File.OpenWrite(@"\SD\" + TextOutputFile);
                System.IO.TextWriter tr = new StreamWriter(fs);

                newMeasurement.TextFileStartPos = fs.Length;
                if (fs.Length == 0)
                {
                    WriteTextFileHeader(aMeasurement, tr);
                }

                fs.Seek(0, SeekOrigin.End);

                WriteTextFileEntry(aMeasurement, tr);
                tr.Flush();
                newMeasurement.TextFileEndPos = fs.Position;
                tr.Close();  
            }
            //GenericMessage.DrawCentered("1.4", true);

            MemoryStream ms = new MemoryStream();
            newMeasurement.ToStream(ms);
            System.IO.File.WriteAllBytes(newMeasurement.Filename, ms.ToArray());

            //GenericMessage.DrawCentered("1.5", true);
            Options.DefaultLot = newMeasurement.Lot;

            Settings.WriteToSDCard("DefaultLot", Options.DefaultLot);
            Settings.WriteToSDCard("DefaultTags", Options.DefaultTags); 

            //GenericMessage.DrawCentered("1.6", true);
            UI.Utils.ExitMenu = true;
            ExitAppMenu = true;
        }

        public virtual void WriteTextFileHeader(Measurement aMeasurement, System.IO.TextWriter tr)
        {
            tr.Write("Lot\tTags\tMeasurement ID\tDevice ID\tReference Scans To Average\tReference Integration Time (ms)\t\tSpecimen Scans To Average\tSpecimen Integration Time (ms)\t");

            tr.Write("\r\n");
        }

        public virtual void WriteTextFileEntry(Measurement aMeasurement, System.IO.TextWriter tr)
        {
            AppMeasurement newMeasurement = (AppMeasurement)aMeasurement;


            tr.Write(newMeasurement.Lot + "\t" +
                        newMeasurement.Tags + "\t" +
                        aMeasurement.MeasurementID.ToString() + "\t" +
                        newMeasurement.DeviceID + "\t" +
                        newMeasurement.ReferenceScansToAverage.ToString() + "\t" +
                        StringUtils.Format(newMeasurement.ReferenceIntegrationTime, 1) + "\t" +
                        newMeasurement.SpecimenScansToAverage.ToString() + "\t" +
                        StringUtils.Format(newMeasurement.SampleIntegrationTime, 1) + "\t");

            tr.Write("\r\n");
        }

        /**
        * This code starts by collecting three samples near zero from the spectrometer.  It then collects additional
        * samples (based on aNumberOfSamples).  Each sample includes the spectrometer's maximum counts and the 
        * integration time used.  The linear slope for these two variables is found for the relationship between
        * counts to integration time.  This slope is used to guess the integration time for the specified target
        * saturation. 
        */
        protected virtual float DetectIntegrationTime(float aTargetSaturation, int aNumberOfSamples, float aZeroCounts, int aScansToAvg)
        {
            int timeout = 50000; //should be 500
            float maxSaturationCounts = 262143f; //The ADC cannot deliver a larger number than this. 

            float[] IntegrationTimes = new float[aNumberOfSamples + 3];
            float[] Maximums = new float[aNumberOfSamples + 3];

            Core.Hardware.Watchdog.Reset();

            //Step 1
            //Find a decent signal low value to start from.  This usually requires 30,000 counts of data.  We start with 5ms.
            IntegrationTimes[0] = 2f;
            Spectrometer.Read(IntegrationTimes[0], aScansToAvg);
            Maximums[0] = MathUtils.Max(Spectrometer.LastSpectra) - (aZeroCounts);

            long startTime = DateTime.Now.Ticks;

            while (Maximums[0] < 30000f)
            {
                IntegrationTimes[0] = 50000f / (Maximums[0] / IntegrationTimes[0]);
                Spectrometer.Read(IntegrationTimes[0], aScansToAvg);
                Maximums[0] = MathUtils.Max(Spectrometer.LastSpectra) - (aZeroCounts);

                if (((DateTime.Now.Ticks - startTime) / TimeSpan.TicksPerMillisecond) > timeout)
                {
                    return -1; //Error occured
                }
            }

            //Step 2
            //Find two more points with low saturation 5% and 10% above the last guess.
            IntegrationTimes[1] = (Maximums[0] * 1.05f) / (Maximums[0] / IntegrationTimes[0]);
            Spectrometer.Read(IntegrationTimes[1], aScansToAvg);
            Maximums[1] = MathUtils.Max(Spectrometer.LastSpectra) - (aZeroCounts);

            IntegrationTimes[2] = (Maximums[0] * 1.10f) / (Maximums[0] / IntegrationTimes[0]);
            Spectrometer.Read(IntegrationTimes[2], aScansToAvg);
            Maximums[2] = MathUtils.Max(Spectrometer.LastSpectra) - (aZeroCounts);

            MathUtils.LinearSlope qeSlope = MathUtils.LinearRegression(Maximums, IntegrationTimes, 0, 3);

            float guess = 10f;  

            float maxTargetCounts = ((maxSaturationCounts - aZeroCounts) * aTargetSaturation);

            float step = maxTargetCounts / IntegrationTimes.Length;


            for (int i = 3; i < IntegrationTimes.Length; i += 1)
            {
                IntegrationTimes[i] = ((step * i) * qeSlope.Slope) + qeSlope.Intercept;

                Spectrometer.Read(IntegrationTimes[i], aScansToAvg);
                Maximums[i] = MathUtils.Max(Spectrometer.LastSpectra) - (aZeroCounts);

                qeSlope = MathUtils.LinearRegression(Maximums, IntegrationTimes, 0, i + 1);

                if (qeSlope.R2 > 0.9f) //When the slope is no longer extreamly linear
                {
                    guess = (((Maximums[i] * aTargetSaturation) - aZeroCounts) * qeSlope.Slope) + qeSlope.Intercept; //Guess based on the user's input 
                }
                else if (i == 3)
                {
                    //Failed on first try.
                    guess = -1;
                }
                else
                {
                    break; //Stop
                }
            }

            return guess;
        }
    }
}