using System;
using Microsoft.SPOT;
using F75XDeviceApp.Common; 
using Core.Utils;
using Core.Hardware;
using System.Threading;
using System.IO;
using Core;
using Core.Text;
using Microsoft.SPOT.IO;
using Core.UI;
using Core.Screens;
using Common;

namespace F75XDeviceApp
{  
    
    public class CustomAppCode : ChemometricsApp
    {
        public class SplashAppMenu : IMenu
        { 
            //Menu's normally
            public int DefaultItem()
            {
                return 1;
            }
             
            private string[] mMenuItems = new string[1] {" "};

            public string[] MenuItems()
            {
                return mMenuItems;
            }
              
            public void OnKeyDown(int aKeyCode, DateTime aTimestamp)
            { 
                if (aKeyCode == KeyPad.KeyCode.Right)
                {
                    MenuController.Setup(new UI.Menus.MainMenu());
                    MenuController.Draw();
                }
                else if (aKeyCode == KeyPad.KeyCode.Up)
                { 
                    CustomAppCode app      = (CustomAppCode)Environment.CurrentApp;
                    string measurementLot  = LotUtils.GetIncrementingPortion(app.Options.DefaultLot, "BIN"); 
                    app.Options.DefaultLot = "\\BIN" + StringUtils.ZeroPadString(int.Parse(measurementLot) + 1, 3) + "\\FRUIT001\\SIDE001";  
                    app.SetupSplashScreen();
                }
                else if (aKeyCode == KeyPad.KeyCode.Down)
                {
                    CustomAppCode app      = (CustomAppCode)Environment.CurrentApp;
                    string measurementLot  = LotUtils.GetIncrementingPortion(app.Options.DefaultLot, "BIN"); 
                    app.Options.DefaultLot = "\\BIN" + StringUtils.ZeroPadString(int.Parse(measurementLot) - 1, 3) + "\\FRUIT001\\SIDE001";  
                    app.SetupSplashScreen();  
                } 
                else if (aKeyCode == KeyPad.KeyCode.Left)
                {  
                     GenericMessage.DrawCentered("Processing...", false);
                     
                    CustomAppCode app = (CustomAppCode)Environment.CurrentApp;
                    
                    string bin   = LotUtils.GetIncrementingPortion(app.Options.DefaultLot, "BIN"); 
                    string fruit = LotUtils.GetIncrementingPortion(app.Options.DefaultLot, "FRUIT");  
                    
                 
                    string lotToErase             = "\\BIN" + StringUtils.ZeroPadString(int.Parse(bin), 3) + "\\FRUIT" + StringUtils.ZeroPadString(int.Parse(fruit), 3);   
                    string pathForLotToEraseSideA = LotUtils.GetFullLotPath("\\BIN" + StringUtils.ZeroPadString(int.Parse(bin), 3) + "\\FRUIT" + StringUtils.ZeroPadString(int.Parse(fruit), 3) + "\\SIDE001");   
                    string pathForLotToEraseSideB = LotUtils.GetFullLotPath("\\BIN" + StringUtils.ZeroPadString(int.Parse(bin), 3) + "\\FRUIT" + StringUtils.ZeroPadString(int.Parse(fruit), 3) + "\\SIDE002");   
                
                    LotUtils.LotStats sideAStats = LotUtils.GetStats(pathForLotToEraseSideA, 1, false);
                     
                    if (sideAStats.RecordCount > 0)
                    {
                        LotUtils.LotStats sideBStats = LotUtils.GetStats(pathForLotToEraseSideB, 1, false);
                        
                        if (sideBStats.RecordCount > 0)
                        { 
                            GenericMessage.DrawCentered("Removing Side B...", false);
                            app.DeleteLot(pathForLotToEraseSideB);  
                            app.Options.DefaultLot = lotToErase + "\\SIDE002"; 
                        }
                        else
                        {
                            GenericMessage.DrawCentered("Removing Side A...", false);
                            app.DeleteLot(pathForLotToEraseSideA);  
                            app.Options.DefaultLot = lotToErase + "\\SIDE001"; 
                        }   
                        app.measurementsPerSide = 1; 
                    } 
                    app.SetupSplashScreen();   
                }
            } 
        }

        public override void ShowSplashScreen()
        { 
            MenuController.Setup(new SplashAppMenu());  
            SetupSplashScreen(); 
        } 
        
        public override void Init()
        {
            base.Init();

            //These are the default options that will be used when this app is first loaded on a device
            Options.DefaultLot                   = "\\BIN001\\FRUIT001\\SIDE001"; //Measurements are organized into lots or groups on the device
            Options.DefaultTags                  = "#MySample #NewUser"; //The Tags for a measurement are essentially metadata.
            Options.SampleScansToAverage         = 32; //The Sample is the fruit (avocado) or target being measured.
            Options.ReferenceScansToAverage      = 32; //The Reference is the reference shutter in the instrument.
            Options.SampleIntegrationTime        = 10; //Integration time in milliseconds, -1 indicates to automatically select the integration time.
            Options.ReferenceIntegrationTime     = 10; //Integration time in milliseconds, -1 indicates to automatically select the integration time.
            Options.EstimateReferenceDarkSpectra = true;
            Options.EstimateSpecimenDarkSpectra  = true;
            DisplayMeasurementWhileSaving        = false;
            
            Options.IncrementationLogic_Lots     = new string[] { "BIN", "FRUIT", "SIDE" };
            Options.IncrementationLogic_Maximums = new int[] { fruitPerBin, sidesPerFruit, measurementsPerSide };
        }
        
        public int fruitPerBin = 100;
        public int sidesPerFruit = 2;
        public int measurementsPerSide = 1;
         
        public override void LoadOptions()
        {
            base.LoadOptions(); 
            Options.SampleScansToAverage         = 32; //The Sample is the fruit (avocado) or target being measured.
            Options.ReferenceScansToAverage      = 32; //The Reference is the reference shutter in the instrument.
            Options.SampleIntegrationTime        = 10; //Integration time in milliseconds, -1 indicates to automatically select the integration time.
            Options.ReferenceIntegrationTime     = 10; //Integration time in milliseconds, -1 indicates to automatically select the integration time.
            Options.EstimateReferenceDarkSpectra = true;
            Options.EstimateSpecimenDarkSpectra  = true; 
        }

        public override void ShowNotes()
        {
            //This clears the screen buffer (drawing canvas in memory), the true (or false) indicates that the cleared screen should be black or white.
            Drawing.ClearScreenBuffer(true);

            //This draws the name of this app (this.Name which is entered under the Overview Tab of App Builder) using the smallest font, at X position 5, Y position 10.  The false or true at the end indicates white or black text.
            SmallFont.DrawString(this.Name, 5, 10, false);
            
            //This draws the unique ID of this app (this.ID which is set by App Builder) using the smallest font, at X position 10, Y position 40.  The false or true at the end indicates white or black text.
            TinyFont.DrawString("ID:" + this.ID, 10, 40, false);
            
            //This draws the date this app was modified using the smallest font, formatted as Month/Day/Year Hour:Minute:Second (see https://tinyurl.com/ycwh45af), at X position 10, Y position 60.  The false or true at the end indicates white or black text.
            TinyFont.DrawString("Modified:" + this.BuildDate.ToString("MM/dd/yyyy HH:mm:ss"), 10, 60, false);

            TinyFont.DrawString("Notes:", 10, 95, false);
            TinyFont.DrawString("This model was created by Felix Instruments.  It is used to", 15, 120, false);
            TinyFont.DrawString("predict using chemometric models.", 15, 140, false);

            SmallFont.DrawString("Press any key to continue...", 5, 210, false);
            
            //Sends the screen buffer (drawing canvas in memory) to display.
            Display.Update(Display.Buffer);
        }
         
        //bool DrawWhileCollecting = true;
        
        /**
         * This code is run when a measurement is saved.  The last line
         * of this method usually displays the measurement.
         */
        public override bool CollectRawSpectra(Measurement aMeasurement)
        { 
            Options.IncrementationLogic_Maximums[2] = measurementsPerSide;
            
            Options.DefaultLot = LotUtils.IncrementLot(
                Options.DefaultLot, 
                Options.IncrementationLogic_Lots, 
                Options.IncrementationLogic_Maximums);
                
              
           // if (DrawWhileCollecting) 
           // {
                aMeasurement.Lot         = Options.DefaultLot; 
                aMeasurement.Filename    = LotUtils.GetNextMeasurementFilename(aMeasurement.Lot);
                string measurementFolder = System.IO.Path.GetDirectoryName(aMeasurement.Filename);
                SetupSplashScreen((AppMeasurement)aMeasurement, measurementFolder, true);
          //  }
            
            return base.CollectRawSpectra(aMeasurement); 
        }
         
        /**
         * This code is run when a measurement is saved. The last line
         * of this method usually displays the measurement.
         */
        public override void SaveMeasurement(Measurement aMeasurement)
        { 
            base.SaveMeasurement(aMeasurement);
             
            string measurementFolder = System.IO.Path.GetDirectoryName(aMeasurement.Filename);
              
            LotUtils.LotStats stats = LotUtils.GetStats(measurementFolder, 1, false);
             
            if (stats.RecordCount >= minimumSideMeasurements)
            {
                if (stats.SD < minimumSideVariance)
                {
                    measurementsPerSide = minimumSideMeasurements;
                }
                else
                {
                    measurementsPerSide += 1;
                }
            }
            else
            {
                measurementsPerSide = minimumSideMeasurements;
            }  
            
            MenuController.Setup(new SplashAppMenu());
            SetupSplashScreen((AppMeasurement)aMeasurement, measurementFolder, false);
        } 
    
        float minimumSideVariance   = 1f;
        int minimumSideMeasurements = 1;
        int maximumSideMeasurements = 1;
        int boxesPerRow             = 1;
        int avocadosPerBin          = 100;
        int boxSize                 = 17;
        int boxPadding              = 5;
        
        public void SetupSplashScreen()
        {      
            string currentlot = Options.DefaultLot;
            
            int bin    = int.Parse(LotUtils.GetIncrementingPortion(Options.DefaultLot, "BIN"));
            for (int fruit = avocadosPerBin; fruit > 0; fruit -= 1)
            {
                for (int side = 2; side > 0; side -= 1)
                {
                    Options.DefaultLot = "\\BIN" + StringUtils.ZeroPadString(bin, 3) + "\\FRUIT" + StringUtils.ZeroPadString(fruit, 3) + "\\SIDE" + StringUtils.ZeroPadString(side, 3);
                    
                    if (System.IO.Directory.Exists(LotUtils.GetFullLotPath(Options.DefaultLot)))
                    {  
                        SetupSplashScreen(null, LotUtils.GetFullLotPath(Options.DefaultLot), false);
                        return;
                    }
                }
            }
            SetupSplashScreen(null, LotUtils.GetFullLotPath(currentlot), false);
        } 
        
        public bool SetupSplashScreen(AppMeasurement measurement, string aMeasurementFolder, bool aPreview)
        {
            int sideA_Count      = 0;
            float sideA_DM       = 0f;
            float sideA_Variance = 0f;

            int sideB_Count      = 0;
            float sideB_DM       = 0f;
            float sideB_Variance = 0f;

            float avocado_DM       = 0f;
            float avocado_Variance = 0f;
            
            int total_Count      = 0;
            float total_DM       = 0f;
            float total_Variance = 0f;
              
            string fruitPath    = LotUtils.GetParentLot(aMeasurementFolder);
            string binPath      = LotUtils.GetParentLot(fruitPath);
            string sideAPath    = fruitPath + "SIDE001\\";
            string sideBPath    = fruitPath + "SIDE002\\";
            
            if (System.IO.Directory.Exists(binPath))
            {    
                LotUtils.LotStats binStats = new LotUtils.LotStats();
                
                binStats = LotUtils.GetStats(binPath, 1, false);
         
                total_Count    = System.IO.Directory.GetDirectories(binPath).Length;
                total_DM       = binStats.Avg;
                total_Variance = binStats.SD;
            }
            
            if (System.IO.Directory.Exists(fruitPath))
            {    
                LotUtils.LotStats fruitStats = new LotUtils.LotStats();
                
                fruitStats = LotUtils.GetStats(fruitPath, 1, false);
          
                avocado_DM       = fruitStats.Avg;
                avocado_Variance = fruitStats.SD;
            }
            
            if (System.IO.Directory.Exists(sideAPath)) 
            {     
                LotUtils.LotStats sideAStats = new LotUtils.LotStats();
                 
                sideAStats     = LotUtils.GetStats(sideAPath, 1, false);
                
                sideA_Count    = sideAStats.RecordCount;
                sideA_DM       = sideAStats.Avg;
                sideA_Variance = sideAStats.SD;
            }
            
            if (System.IO.Directory.Exists(sideBPath)) 
            {     
                LotUtils.LotStats sideBStats = new LotUtils.LotStats();
                 
                sideBStats = LotUtils.GetStats(sideBPath, 1, false);
         
                sideB_Count    = sideBStats.RecordCount;
                sideB_DM       = sideBStats.Avg;
                sideB_Variance = sideBStats.SD;
            }  
 

            //int fruitNumber = int.Parse(LotUtils.GetIncrementingPortion(aMeasurementFolder, "FRUIT"));   && (fruitNumber == avocadosPerBin)
            
            bool sideADone = ((sideA_Count >= minimumSideMeasurements) && (sideA_Variance <= minimumSideVariance));
            bool sideBDone = ((sideB_Count >= minimumSideMeasurements) && (sideB_Variance <= minimumSideVariance));
            bool fruitDone = (sideADone && sideBDone);
         
            DrawAppSplashScreen(measurement, aMeasurementFolder,
            sideADone, sideA_Count, sideA_DM, sideA_Variance,
            sideBDone, sideB_Count, sideB_DM, sideB_Variance,
            fruitDone, avocado_DM, avocado_Variance,
            total_Count, total_DM, total_Variance, aPreview);
            
            return fruitDone;
        }
        
        bool AutoRepeatMeasurements = false;
        
        public void DrawAppSplashScreen(AppMeasurement measurement,
            string aMeasurementFolder,
            bool sideADone, int sideA_Count, float sideA_DM, float sideA_Variance,
            bool sideBDone, int sideB_Count, float sideB_DM, float sideB_Variance,
            bool fruitDone, float avocado_DM, float avocado_Variance,
            int total_Count, float total_DM, float total_Variance, bool aPreview)
        {
            
            bool invertScreen = true;

            bool hideUnusedBoxes = true;

            Drawing.ClearScreenBuffer(!invertScreen);
            
            //TinyFont.DrawString(LotUtils.GetRelativeLotPath(aMeasurementFolder), 145, 5, false);

            //Draws the black region on the left
            Drawing.DrawRectangle(0, 0, 175, 240, true, Drawing.PixelColor.Black);

            //Side A
            SmallFont.DrawString("Side A", 12, 30, false);
            
            if (sideADone)
            {
                if (hideUnusedBoxes)
                {
                    DrawBoxes(boxPadding, boxSize, 12, 80, sideA_Count, sideA_Count, sideA_Count, false);
                }
                else
                {
                    DrawBoxes(boxPadding, boxSize, 12, 80, boxesPerRow, boxesPerRow, boxesPerRow, false);
                }
            }
            else
            {
                DrawBoxes(boxPadding, boxSize, 12, 80, sideA_Count, boxesPerRow, boxesPerRow, aPreview);
            }
             
            if (sideA_Count > 0)
            {
                SmallFont.DrawString(Core.Utils.StringUtils.Format(sideA_DM, 1) + "%", 12, 54, false);
            }

            //Side B
            SmallFont.DrawString("Side B", 100, 30, false);
            
            if (sideBDone)
            {
                if (hideUnusedBoxes)
                {
                    DrawBoxes(boxPadding, boxSize, 100, 80, sideB_Count, sideB_Count, sideB_Count, false); 
                }
                else 
                {
                    DrawBoxes(boxPadding, boxSize, 100, 80, boxesPerRow, boxesPerRow, boxesPerRow, false); 
                }
            }
            else 
            {
                if (!sideADone)
                {                    
                    DrawBoxes(boxPadding, boxSize, 100, 80, 0, boxesPerRow, boxesPerRow, false);
                }
                else 
                {
                    DrawBoxes(boxPadding, boxSize, 100, 80, sideB_Count, boxesPerRow, boxesPerRow, aPreview);
                }
            }

            
            if (sideB_Count > 0)
            {
                SmallFont.DrawString(Core.Utils.StringUtils.Format(sideB_DM, 1) + "%", 100, 54, false);
            }
            
            
            if (AutoRepeatMeasurements)
            {
                if ((sideA_Count > 0) && (!sideADone))
                { 
                    RepeatMeasurement = true;
                }
                else if ((sideB_Count > 0) && (!sideBDone))
                {
                    RepeatMeasurement = true;
                }
                else
                {
                    RepeatMeasurement = false;
                }
            }
             
             
            //Avocado
            SmallFont.DrawString("Current Bin", 12, 130, false);
            
            if (total_Count > 0)
            {
                SmallFont.DrawString("Avg.", 12, 158, false);
                SmallFont.DrawString(Core.Utils.StringUtils.Format(total_DM, 1) + "%", 75, 158, false);
                SmallFont.DrawString("Avocado # " + Core.Utils.StringUtils.Format(total_Count, 0), 12, 6, false);
                if (total_Variance > 0)
                {
                    SmallFont.DrawString("Variance", 12, 186, false);
                    DrawPlusOrMinus(110, 191, 12, 2, true);
                    SmallFont.DrawString(Core.Utils.StringUtils.Format(total_Variance, 1), 125, 186, false);
                }
            } 

            string relativeLot = LotUtils.GetRelativeLotPath(aMeasurementFolder).Split('\\')[0];
            
            SmallFont.DrawString("\\" + relativeLot.ToUpper(), 197, 6, true);
            
            if (((sideA_Count == maximumSideMeasurements) && !sideADone) || ((sideB_Count == maximumSideMeasurements) && !sideBDone))
            {
                RepeatMeasurement = false;
                
                LargeFont.DrawString("Error", 157, 70, true);
                
                SmallFont.DrawString("Side Variation >2%", 197, 145, true);
                SmallFont.DrawString("Remove side and", 197, 175, true);
                SmallFont.DrawString("retry or select", 197, 195, true);
                SmallFont.DrawString("a new bin.", 197, 215, true);
            }
            else if (sideB_Count > 0 && (sideA_DM * 1.1 < sideB_DM || sideA_DM * 0.9 > sideB_DM))
            {
                SmallFont.DrawString("Warning!", 197, 60, true);
                SmallFont.DrawString("Side B Dry Matter", 197, 85, true);
                SmallFont.DrawString("is significantly", 197, 110, true);
                SmallFont.DrawString("different from", 197, 135, true);
                SmallFont.DrawString("Side A and may not", 197, 160, true);
                SmallFont.DrawString("be an accurate", 197, 185, true);
                SmallFont.DrawString("measurement.", 197, 210, true);
            }
            else if (sideA_Count > 0)
            {
                LargeFont.DrawString(Core.Utils.StringUtils.Format(avocado_DM, 1) + "%", 192, 70, true);   
                SmallFont.DrawString("Avocado Dry Matter", 197, 135, true); 
                if (avocado_Variance > 0)
                {
                    DrawPlusOrMinus(197, 170, 12, 2, false);
                    SmallFont.DrawString(Core.Utils.StringUtils.Format(avocado_Variance, 1), 212, 165, true); 
                }
            }
            else if (aPreview)
            {
                LargeFont.DrawString("Wait...", 192, 70, true);
                SmallFont.DrawString("Keep instrument", 197, 145, true);
                SmallFont.DrawString("and avocado", 197, 165, true);
                SmallFont.DrawString("steady.", 197, 185, true);
            }
            else
            { 
                
                LargeFont.DrawString("Empty", 192, 70, true);  
                
                SmallFont.DrawString("Measure the ventral", 197, 145, true);
                SmallFont.DrawString("and dorsal sides", 197, 165, true);
                SmallFont.DrawString("of 100 avocados", 197, 185, true);
                SmallFont.DrawString("to fill bin.", 197, 205, true);
                
                /*SmallFont.DrawString("Measure ventral (A)", 162, 75, true);    
                SmallFont.DrawString("and dorsal (B) sides", 162, 95, true);    
                SmallFont.DrawString("of 10 avocado to", 162, 115, true);      
                SmallFont.DrawString("complete each bin.", 162, 135, true);    
                TinyFont.DrawString("Up/Down changes bins", 162, 170, true);     
                TinyFont.DrawString("Left removes last side", 162, 190, true);     
                TinyFont.DrawString("Right opens main menu", 162, 210, true);  */
            }
            
            //
            
            

            UpdateBatteryLevel(true);

            Display.Update(Display.Buffer); 
        }
        
        
        private void DrawBoxes(int paddingBetweenSquares, int squareSize, int squareOffsetX, int squareOffsetY, int solidSquares, int totalSquares, int squareColumns, bool drawIncomplete)
        {
            int currentSquareColumn = 0;
            int currentSquareRow = 0;

            for (int i = 0; i < totalSquares; i += 1)
            {

                if (currentSquareColumn == squareColumns)
                { 
                    currentSquareRow += 1;
                    currentSquareColumn = 0;
                } 

                int currentSquareX = squareOffsetX + ((squareSize + paddingBetweenSquares) * currentSquareColumn);
                int currentSquareY = squareOffsetY + ((squareSize + paddingBetweenSquares) * currentSquareRow);

                if ((drawIncomplete) && (i == solidSquares))
                { 
                    Drawing.DrawRectangle(currentSquareX, currentSquareY, squareSize, squareSize, false, Drawing.PixelColor.White);
                    Drawing.DrawRectangle(currentSquareX + (int)(squareSize * 0.5f), currentSquareY, (int)(squareSize * 0.25f), (int)(squareSize * 0.25f), true, Drawing.PixelColor.White); 
                    Drawing.DrawRectangle(currentSquareX, currentSquareY, (int)(squareSize * 0.5f), (int)(squareSize * 0.5f), true, Drawing.PixelColor.White); 
                    Drawing.DrawRectangle(currentSquareX, currentSquareY + (int)(squareSize * 0.5f), (int)(squareSize * 0.25f), (int)(squareSize * 0.25f), true, Drawing.PixelColor.White); 
                }  
                else if (i < solidSquares)
                { 
                    Drawing.DrawRectangle(currentSquareX, currentSquareY, squareSize, squareSize, true, Drawing.PixelColor.White);
                }
                else
                { 
                    Drawing.DrawRectangle(currentSquareX, currentSquareY, squareSize, squareSize, false, Drawing.PixelColor.White);
                } 

                currentSquareColumn += 1;
            }
        }
        
        private void DrawPlusOrMinus(int aX, int aY, int aWidth, int thickness, bool black)
        { 
            if (!black) 
            {
                Drawing.DrawRectangle((int)(aX + (aWidth / 2f) - (thickness / 2f)), aY, thickness, aWidth, true, Drawing.PixelColor.Black);
                Drawing.DrawRectangle(aX, (int)(aY + (aWidth / 2f) - (thickness / 2f)), aWidth, thickness, true, Drawing.PixelColor.Black);
                Drawing.DrawRectangle(aX, aY + aWidth + 2, aWidth, thickness, true, Drawing.PixelColor.Black);
            }
            else
            {
                Drawing.DrawRectangle((int)(aX + (aWidth / 2f) - (thickness / 2f)), aY, thickness, aWidth, true, Drawing.PixelColor.White);
                Drawing.DrawRectangle(aX, (int)(aY + (aWidth / 2f) - (thickness / 2f)), aWidth, thickness, true, Drawing.PixelColor.White);
                Drawing.DrawRectangle(aX, aY + aWidth + 2, aWidth, thickness, true, Drawing.PixelColor.White);
            } 
        }

        public void UpdateBatteryLevel(bool aUpdateBattery)
        { 
            double batLevel;
 
            if (aUpdateBattery)
            {
                batLevel = Core.Hardware.Battery.GetUpdatedLevel();
            }
            else
            {
                batLevel = Core.Hardware.Battery.CurrentLevel;
            }

            Drawing.DrawRectangle(328 + 7, 13 - 7, 22, 12, false, Drawing.PixelColor.Black);
            Drawing.DrawRectangle(330 + 7, 15 - 7, 18, 8, true, Drawing.PixelColor.White);

            int max = 18;
            int min = 1;
            int value = 1 + System.Math.Max(System.Math.Min((int)(min + (((max - min) / 100f) * batLevel)), 100), 1);

            Drawing.DrawRectangle(330 + 7, 15 - 7, value, 8, true, Drawing.PixelColor.Black);

            Drawing.DrawRectangle(350 + 7, 15 - 7, 2, 8, false, Drawing.PixelColor.Black);

            if (batLevel >= 100)
            {
                batLevel = 99;
            }

            TinyFont.DrawString(ref Display.Buffer, (int)batLevel + "%", 362 + 3, 12 - 7, true);

        }
        
        public override void BuildSpectra(Measurement aMeasurement) 
        {
            base.BuildSpectra(aMeasurement); //Runs default code
        }

        /**
         * This code is run on the PC when building chemometric models and
         * also on the device when new measurements are recorded.
         */
        public override void SetPredictions(Measurement aMeasurement)
        {
            //Uncomment the example code below to set the first 
            //UserDefinedValues field to "Yes" or "No" based on if
            //predictions were modified.
            //
            //Example code... 
            //
            //UserDefinedValues = new string[1];
            //
            //if (aMeasurement.PredictionValue[0] >= 1) 
            //{
            //     aMeasurement.PredictionValue[0] = 1;
            //     UserDefinedValues[0] = "Yes";
            //}
            //else if (aMeasurement.PredictionValue[0] <= 0) 
            //{
            //     aMeasurement.PredictionValue[0] = 0;
            //     UserDefinedValues[0] = "Yes";
            //}
            //else
            //{ 
            //     UserDefinedValues[0] = "No";
            //}

        
            base.SetPredictions(aMeasurement); //Runs default code
            
            AppMeasurement appMeasurement = (AppMeasurement)aMeasurement;

            Random randomGenerator = new Random(); 
            appMeasurement.PredictionValues[0] = randomGenerator.Next();
        } 
 
        /**
         * Writes the header for the output text file on the device.
         */
        public override void WriteTextFileHeader(Measurement aMeasurement, System.IO.TextWriter tr)
        { 
            //Uncomment the example code below to add the column WasAdjusted
            //to the output text file.
            //
            //Example code...
            //tr.Write("WasAdjusted\t");

            base.WriteTextFileHeader(aMeasurement, tr); //Runs default code
            
        }
        
        /**
         * Writes a line for new measurements in the output text file on the device.
         */
        public override void WriteTextFileEntry(Measurement aMeasurement, System.IO.TextWriter tr)
        {
            //Uncomment the example code below to add the first UserDefinedValue
            //Example code...
            //
            //AppMeasurement newMeasurement = (AppMeasurement)aMeasurement;
            //tr.Write((newMeasurement.UserDefinedValues[0]));

            base.WriteTextFileEntry(aMeasurement, tr); //Runs default code 
        }
    }
}